// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/io/string_table.hh"

nv::string_table_creator::string_table_creator()
{
	insert(""); // 0 always is empty string
}

nv::string_table_creator::index nv::string_table_creator::insert( const std::string& s )
{
	auto i = m_map.find( s );
	if ( i != m_map.end() )
	{
		return i->second;
	}
	const char* cs = s.c_str();
	uint32 cs_size = s.size() + 1;
	NV_ASSERT( m_offsets.size() < index(-1), "Too many strings!" );
	index  result  = (index)m_offsets.size();
	size_t dsize = m_data.size();
	m_offsets.push_back( dsize );
	m_data.resize( dsize + cs_size );
	raw_copy( cs, cs + cs_size, m_data.data() + dsize );
	m_map[ s ] = result;
	return result;
}

nv::string_table* nv::string_table_creator::create_table() const
{
	offset* offsets = new offset[m_offsets.size()];
	char*   data    = new char [m_data.size()];
	raw_copy( m_offsets.begin(), m_offsets.end(), offsets );
	raw_copy( m_data.begin(),    m_data.end(),    data );
	return new string_table( data, m_data.size(), offsets, (index)m_offsets.size() );
}

void nv::string_table_creator::dump( nv::stream* out ) const
{
	index  count = (index)m_offsets.size();
	uint32 size  = m_data.size();
	out->write( &count,  sizeof( count ), 1 );
	out->write( &size,   sizeof( size ), 1 );
	out->write( m_offsets.data(), sizeof( offset ), count );
	out->write( m_data.data(),    sizeof( char ),  size );
}

const char* nv::string_table_creator::get( index i ) const
{
	return i < m_offsets.size() ? m_data.data() + m_offsets[i] : nullptr;
}

nv::string_table_creator::index nv::string_table_creator::get( const std::string& s ) const
{
	auto i = m_map.find( s );
	if ( i != m_map.end() )
	{
		return i->second;
	}
	return 0;
}

void nv::string_table_creator::clear()
{
	m_map.clear();
	m_offsets.clear();
	m_data.clear();
}

nv::uint32 nv::string_table_creator::dump_size() const
{
	return sizeof( index ) + sizeof( uint32 ) +
		sizeof( offset ) * m_offsets.size() +
		sizeof( char ) * m_data.size();
}

