// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file contiguous_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief contiguous_storage classes
*/

#ifndef NV_STL_CONTAINER_CONTIGUOUS_STORAGE_HH
#define NV_STL_CONTAINER_CONTIGUOUS_STORAGE_HH

#include <nv/core/common.hh>
#include <nv/stl/type_traits/alignment.hh>
#include <nv/stl/capi.hh>

namespace nv
{

	template< typename T, size_t N >
	class static_storage
	{
	public:
		typedef T      value_type;

		static constexpr bool is_static = true;
		static constexpr bool is_const = false;

		constexpr const value_type* data() const { return reinterpret_cast<const value_type*>( m_data ); }
		inline    value_type* data() { return reinterpret_cast<T*>( m_data ); }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }

	protected:
		static constexpr bool reallocate( size_t new_size, bool /*copy_needed*/ ) { return new_size <= N; }

		static_storage() = default;

		// prevent copying 
		static_storage( const static_storage& ) = delete;
		static_storage& operator=( const static_storage& ) = delete;
		// allow move
		static_storage( static_storage&& other ) = default;
		static_storage& operator=( static_storage&& other ) = default;

		~static_storage() = default;
	protected:
		typedef aligned_array_t<T, N, alignof( T ) > storage_type;
		storage_type m_data;
	};
	
	template< typename T >
	class dynamic_storage
	{
	public:
		typedef T      value_type;

		static constexpr bool is_static = false;
		static constexpr bool is_const = false;

		constexpr const value_type* data() const { return reinterpret_cast<const value_type*>( m_data ); }
		inline    value_type* data() { return reinterpret_cast<T*>( m_data ); }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }

	protected:
		constexpr dynamic_storage() : m_data( nullptr ) {}
		// prevent copying 
		dynamic_storage( const dynamic_storage& ) = delete;
		dynamic_storage& operator=( const dynamic_storage& ) = delete;
		// allow move
		inline dynamic_storage( dynamic_storage&& other )
			: m_data( other.m_data )
		{
			other.m_data = nullptr;
		}
		inline dynamic_storage& operator=( dynamic_storage&& other )
		{
			if ( this != &other )
			{
				reallocate( 0, false );
				m_data = other.m_data;
			}
			return *this;
		}
		~dynamic_storage() = default;

		bool reallocate( size_t new_size, bool copy_needed )
		{
			if ( copy_needed )
				m_data = (uint8*)nvrealloc( m_data, new_size * sizeof( value_type ) );
			else
			{
				nvfree( m_data );
				m_data = ( new_size > 0 ? (uint8*)nvmalloc( new_size * sizeof( value_type ) ) : nullptr );
			}
			return true; // TODO : alloc check?
		}
	protected:
		uint8* m_data;
	};

}

#endif // NV_STL_CONTIGUOUS_STORAGE_HH

