// Copyright (C) 2012-2014 ChaosForge Ltd
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: 
//  * performance tests
//  * matching tests
//  * compatibility tests
//  * check if we can get rid of const templates
//
// String reference implementation based of string_ref proposal and boost::string_ref
//
// http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2012/n3442.html
// http://www.boost.org/doc/libs/1_58_0/libs/utility/doc/html/string_ref.html
//
// TODO: WARNING: this code is centered around 8bit char. In particular, 
//   const_string wont work with non-8bit chars.
// TODO: remove templating on char type, make a string_utf8 instead.

#ifndef NV_CORE_STRING_HH
#define NV_CORE_STRING_HH

#include <string>
#include <sstream>
#include <nv/core/common.hh>
#include <nv/stl/type_traits/primary.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/exception.hh>
#include <nv/stl/algorithm.hh>

namespace nv
{
	/**
	* Simple function for slurping a file into a string.
	*/
	std::string slurp( const std::string& filename );

	namespace detail
	{
		// These could be done much better
		template <typename T>
		struct is_cstring_impl
		{
			typedef conditional_t < is_array<T>::value, remove_extent_t<T>*, remove_cv_t<T>	> decayed_type;
			typedef bool_constant <	
				is_same<       char *, decayed_type >::value ||	
				is_same< const char *, decayed_type >::value > type;
		};

	}

	template < typename T >
	struct is_cstring : detail::is_cstring_impl< remove_reference_t<T> >::type
	{

	};
	
	class string_ref;

	// string base class - will become a base for a string class later
	class string_base : public detail::add_iterators< const_storage_view< char > >
	{
		typedef detail::add_iterators< const_storage_view< char > > inherited;
	public:
		typedef char           value_type;
		typedef const char*    pointer;
		typedef const char&    reference;
		typedef const char&    const_reference;
		typedef pointer        const_iterator;
		typedef const_iterator iterator;
		typedef size_t         size_type;
		typedef ptrdiff_t      difference_type;

		static constexpr size_type npos = size_type( -1 );

		// conversion to std::string
 		inline std::string to_string() const
 		{
 			return std::string( data(), size() );
 		}

		inline size_type length()   const { return size(); }

		// access
		inline char operator[]( size_type i ) const { return data()[i]; }
		inline char at( size_type i ) const
		{
			//	if ( i >= m_data ) NV_THROW( out_of_range( "string_ref::at" ) );
			return data()[i];
		}

		inline char front()        const { return data()[0]; }
		inline char back()         const { return data()[size() - 1]; }

		// string operations
		int compare( const string_base& rhs ) const
		{
			size_type this_size = size();
			int cmp = nvmemcmp( data(), rhs.data(), ( nv::min )( this_size, rhs.size() ) );
			return cmp != 0 ? cmp : ( this_size == rhs.size() ? 0 : this_size < rhs.size() ? -1 : 1 );
		}
		bool starts_with( char c ) const { return !empty() && c == front(); }
		bool starts_with( const string_base& s ) const
		{
			return size() >= s.size() && nvmemcmp( data(), s.data(), s.size() ) == 0;
		}
		bool ends_with( char c ) const { return !empty() && c == back(); }
		bool ends_with( const string_base& s ) const
		{
			return size() >= s.size() && nvmemcmp( data() + size() - s.size(), s.data(), s.size() ) == 0;
		}
		size_type find( const string_base& s, size_type pos = 0 ) const
		{
			if ( pos >= size() ) return npos;
			const_iterator it = search( this->cbegin() + ( difference_type ) pos, this->cend(), s.cbegin(), s.cend() );
			return it == this->cend() ? npos : ( size_type )distance( this->cbegin(), it );
		}
		size_type find( char c, size_type pos = 0 ) const
		{
			if ( pos >= size() ) return npos;
			const_iterator it = find_if( this->cbegin() + (difference_type)pos, this->cend(), [=] ( char val ) { return val == c; } );
			return it == this->cend() ? npos : ( size_type )distance( this->cbegin(), it );
		}
		size_type rfind( const string_base& s, size_type pos = 0 ) const
		{
			if ( pos >= size() ) return npos;
			const_reverse_iterator it = search( this->crbegin() + (difference_type)pos, this->crend(), s.crbegin(), s.crend() );
			return it == this->crend() ? npos : reverse_distance( this->crbegin(), it );
		}
		size_type rfind( char c, size_type pos = 0 ) const
		{
			if ( pos >= size() ) return npos;
			const_reverse_iterator it = find_if( this->crbegin() + (difference_type)pos, this->crend(), [=] ( char val ) { return val == c; } );
			return it == this->crend() ? npos : reverse_distance( this->crbegin(), it );
		}
		size_type find_first_of( char c ) const { return find( c ); }
		size_type find_first_of( const string_base& s ) const
		{
			const_iterator it = nv::find_first_of( this->cbegin(), this->cend(), s.cbegin(), s.cend() );
			return it == this->cend() ? npos : ( size_type )distance( this->cbegin(), it );
		}
		size_type find_last_of( char c )  const { return rfind( c ); }
		size_type find_last_of( const string_base& s ) const
		{
			const_reverse_iterator it = nv::find_first_of( this->crbegin(), this->crend(), s.cbegin(), s.cend() );
			return it == this->crend() ? npos : reverse_distance( this->crbegin(), it );
		}
		size_type find_first_not_of( const string_base& s ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( 0 == nvmemchr( s.data(), (uchar8)*it, s.size() ) )
					return ( size_type )distance( this->cbegin(), it );
			return npos;
		}
		size_type find_first_not_of( char c ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( c != *it )
					return ( size_type )distance( this->cbegin(), it );
			return npos;
		}
		size_type find_last_not_of( const string_base& s ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( 0 == nvmemchr( s.data(), (uchar8)*it, s.size() ) )
					return reverse_distance( this->crbegin(), it );
			return npos;
		}
		size_type find_last_not_of( char c ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( c != *it )
					return reverse_distance( this->crbegin(), it );
			return npos;
		}

		// string operations
		string_ref substr( size_type p, size_type n = npos ) const;

		inline size_t hash() const
		{
			const char* str = data();
			size_type   sz  = size();
			int seed = 131;
			int result = 0;
			while ( sz )
			{
				result = ( result * seed ) + ( *str );
				str++;
				sz--;
			}
			return result & ( 0x7FFFFFFF );
		}

		// Literal constructors
		template< size_t N >
		inline string_base( char( &s )[N] ) : inherited( s, N - 1 ) {}
		template< size_t N >
		inline string_base( const char( &s )[N] ) : inherited( s, N - 1 ) {}

	protected:
		inline string_base() {}
		inline string_base( pointer a_data, size_type a_lenght ) : inherited( a_data, a_lenght ) {}

		template < typename ReverseIterator >
		size_type reverse_distance( ReverseIterator first, ReverseIterator last ) const
		{
			return size() - 1 - (size_t)distance( first, last );
		}
	};

	class string_ref : public string_base
	{
	public:
		inline string_ref() {}
		inline string_ref( const string_ref& rhs )
			: string_base( rhs.data(), rhs.size() )
		{
		}
		inline string_ref( const string_base& rhs )
			: string_base( rhs.data(), rhs.size() )
		{
		}

		inline string_ref( const std::string& str )
			: string_base( str.data(), str.size() )
		{
		}

		inline string_ref( const char* str, size_type len )
			: string_base( str, len )
		{
		}

		// Literal constructors
		template< size_t N >
		inline string_ref( char( &s )[N] ) : string_base( s, N - 1 ) {}
		template< size_t N >
		inline string_ref( const char( &s )[N] ) : string_base( s, N - 1 ) {}

		// Non-literal constructors
		template< typename U >
		inline string_ref( U str, typename enable_if<is_same<U, const char*>::value>::type* = nullptr ) : string_base( str, nvstrlen( str ) ) {}

		// Non-literal constructors
		template< typename U >
		inline string_ref( U str, typename enable_if<is_same<U, char*>::value>::type* = nullptr ) : string_base( str, nvstrlen( str ) ) {}

		inline string_ref& operator=( const string_ref &rhs )
		{
			assign( rhs.data(), rhs.size() );
			return *this;
		}

		// modifiers
		inline void clear()
		{
			assign( nullptr, 0 );
		}
		inline void remove_prefix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data() + n, s - n );
		}
		inline void remove_suffix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data(), s - n );
		}

	};

	// const string is movable but not copyable
	class const_string : public string_base
	{
	public:
		inline const_string() {}
		inline const_string( const char* str, size_type len )
		{
			initialize( str, len );
		}
		inline explicit const_string( const char* str )
		{
			initialize( str, nvstrlen( str ) );
		}

		~const_string()
		{
			if ( data() )
			{
				delete data();
			}
		}

		inline const_string( const_string&& other )
		{
			assign( other.data(), other.size() );
			other.assign( nullptr, 0 );
		}

		inline const_string& operator=( const_string&& other )
		{
			pointer   old_data = data();
			size_type old_size = size();
			assign( other.data(), other.size() );
			other.assign( old_data , old_size );
			return *this;
		}

		// blocked copy constructor
		const_string( const const_string & ) = delete;
		// blocked copy operator
		const_string& operator=( const const_string& ) = delete;

	private:

		void initialize( const char* p, size_type s )
		{
			char* data = new char[s + 1];
			nvmemcpy( data, p, s );
			data[s] = 0;
			assign( data, s );
		}
	};

	inline string_ref string_base::substr( size_type p, size_type n ) const
	{
		if ( p > size() ) return string_ref(); // NV_THROW( out_of_range( "substr" ) );
		if ( n == p || p + n > size() ) n = size() - p;
		return string_ref( data() + p, n );
	}

#define NV_STRING_BASE_CAST_OPERATORS( OPERATOR )\
inline bool operator OPERATOR ( const string_base& lhs, const std::string& rhs )\
{\
	return lhs OPERATOR string_ref( rhs );\
}\
inline bool operator OPERATOR ( const std::string& lhs, const string_base& rhs )\
{\
	return string_ref( lhs ) OPERATOR rhs;\
}\
inline bool operator OPERATOR ( const string_base& lhs, const char* rhs )\
{\
	return lhs OPERATOR string_ref( rhs );\
}\
inline bool operator OPERATOR ( const char* lhs, const string_base& rhs )\
{\
	return string_ref( lhs ) OPERATOR rhs;\
}\

	// Base operators
	inline bool operator==( const string_base& lhs, const string_base& rhs )
	{
		return lhs.size() != rhs.size() ? false : ( lhs.compare( rhs ) == 0 );
	}
	inline bool operator!=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.size() != rhs.size() ? true : ( lhs.compare( rhs ) != 0 );
	}
	inline bool operator<( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) < 0;
	}
	inline bool operator>( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) > 0;
	}
	inline bool operator<=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) <= 0;
	}
	inline bool operator>=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) >= 0;
	}

NV_STRING_BASE_CAST_OPERATORS( == )
NV_STRING_BASE_CAST_OPERATORS( != )
NV_STRING_BASE_CAST_OPERATORS( < )
NV_STRING_BASE_CAST_OPERATORS( > )
NV_STRING_BASE_CAST_OPERATORS( <= )
NV_STRING_BASE_CAST_OPERATORS( >= )

	inline std::ostream& operator<<( std::ostream& os, const string_base& str )
	{
		if ( os.good() )
		{
			os.write( str.data(), static_cast<std::streamsize>( str.size() ) );
		}
		return os;
	}

#undef NV_STRING_REF_CAST_OPERATORS

size_t sint32_to_buffer( sint32 n, char* str );
size_t sint64_to_buffer( sint64 n, char* str );
size_t uint32_to_buffer( uint32 n, char* str );
size_t uint64_to_buffer( uint64 n, char* str );
size_t f32_to_buffer( f32 n, char* str );
size_t f64_to_buffer( f64 n, char* str );

inline string_ref trimmed( const string_ref& str )
{
	size_t endpos = str.find_last_not_of( " \r\n\t" );
	size_t startpos = str.find_first_not_of( " \r\n\t" );

	if ( string_ref::npos != endpos || string_ref::npos != startpos )
	{
		if ( string_ref::npos == startpos ) startpos = 0;
		if ( string_ref::npos != endpos )   endpos = endpos + 1 - startpos;
		return str.substr( startpos, endpos );
	}
	return str;
}

inline string_ref rtrimmed( const string_ref& str )
{
	size_t endpos = str.find_last_not_of( " \r\n\t" );
	if ( string_ref::npos != endpos )
	{
		return str.substr( 0, endpos + 1 );
	}
	return str;
}

inline string_ref ltrimmed( const string_ref& str )
{
	size_t startpos = str.find_first_not_of( " \r\n\t" );
	if ( string_ref::npos != startpos )
	{
		return str.substr( startpos );
	}
	return str;
}


inline string_ref extract_extension( string_ref filename )
{
	size_t lastdot = filename.find_last_of( '.' );
	if ( string_ref::npos != lastdot )
		return filename.substr( lastdot + 1 );
	return filename;
}


}

#endif // NV_CORE_STRING_HH
