// Copyright (C) 2014 Kornel Kisielewicz
// This file is part of Nova Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_CORE_TYPES_HH
#define NV_CORE_TYPES_HH

#include <nv/core/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/cstring_store.hh>
#include <nv/stl/type_traits/properties.hh>
#include <unordered_map>
#include <vector>

namespace nv
{
	typedef nv::uint64 type_hash;

	template< typename T >
	struct is_container
	{
	private:
		typedef char                      yes;
		typedef struct { char array[2]; } no;
		template<typename C> static yes test( typename C::iterator* );
		template<typename C> static no  test( ... );
	public:
		static const bool value = sizeof( test<T>( 0 ) ) == sizeof( yes );
	};

	template<>
	struct is_container < std::string >
	{
		static const bool value = false;
	};

	struct type_entry;
	class type_database;

	enum type_flag
	{
		TF_POINTER     = 0x01, //!< field is a pointer
		TF_NOSERIALIZE = 0x02, //!< ignore during serialization
		TF_INVISIBLE   = 0x04, //!< field invisible to API
		TF_READONLY    = 0x08, //!< read only field
		TF_SIMPLETYPE  = 0x10, //!< raw binary I/O possible
		TF_OWNED       = 0x20,
		TF_CONTAINER   = 0x40, //!< is a container
		TF_INDEXED     = 0x80, //!< type is a index to another type
	};

	struct type_field
	{
		uint32      name;     //!< name of the field
		type_hash   hash; //!< typeinfo for later retrieval of type
		type_entry* type;     //!< pointer to field type
		uint32      flags;    //!< flags 
		uint32      offset;   //!< offset into parent
	};

	struct type_enum
	{
		uint32 name;
		sint32 value;
	};

	struct type_entry
	{
		// Function types for the constructor and destructor of registered types
		typedef void( *constructor_func )( void* );
		typedef void( *destructor_func )( void* );

		type_database*          type_db;     //!< Parent type database
		uint32                  name;        //!< Scoped C++ name of the type
		constructor_func        constructor; //!< Pointers to the constructor 
		destructor_func         destructor;  //!< Pointers to the destructor 
		size_t                  size;        //!< Result of sizeof(type) operation
		type_entry*             base_type;   //!< Base type
		std::vector<type_field> field_list;  //!< Field list
		std::vector<type_enum>  enum_list;   //!< Enum list
		cstring_store names;
	};

	class type_creator
	{
	public:
		type_creator( type_database* db, type_entry* entry )
			: m_database( db ), m_entry( entry ) {}

		template <typename TYPE>
		type_creator base();

		template< typename TOBJECT, typename TFIELD >
		type_creator field( const char* aname, TFIELD TOBJECT::*field, typename enable_if< is_container<TFIELD>::value, void* >::type = nullptr );

		template< typename TOBJECT, typename TFIELD >
		type_creator field( const char* aname, TFIELD TOBJECT::*field, typename enable_if< !is_container<TFIELD>::value, void* >::type = nullptr );

		type_creator value( const char* name, sint32 value );
	private:
		type_database* m_database;
		type_entry*    m_entry;
	};

	class type_database
	{
	public:
		template< typename TYPE >
		type_creator create_type( const char* name )
		{
			NV_ASSERT( !m_names.exists( name ), "Type redefinition!" );
			uint32 name_idx = m_names.push( name );
			type_entry* i_type = new type_entry;
			i_type->type_db = this;
			i_type->name = name_idx;
			i_type->size = sizeof( TYPE );

			i_type->constructor = raw_construct_object < TYPE >;
			i_type->destructor  = raw_destroy_object < TYPE >;
			m_idx_types[typeid( TYPE )] = i_type;
			m_type_list.push_back( i_type );
			return type_creator( this, i_type );
		}

		type_entry* get_type( const char* name )
		{
			uint32 name_idx = m_names.resolve( name );
			if ( name_idx < m_type_list.size() )
			{
				return m_type_list[name_idx];
			}
			return nullptr;
		}

		type_entry* get_type( type_hash hash )
		{
			type_info_map::iterator it = m_idx_types.find( hash );
			if ( it != m_idx_types.end() )
			{
				return it->second;
			}
			return nullptr;
		}

		const char* resolve_name( uint32 name_idx )
		{
			return m_names.get( name_idx );
		}

		~type_database()
		{
			for ( auto t : m_type_list ) delete t;
		}
	private:
		typedef std::vector<type_entry*>                   type_list;
		typedef std::unordered_map<type_hash, type_entry*> type_info_map;
		cstring_store m_names;
		type_list     m_type_list;
		type_info_map m_idx_types;
	};
	
	template < typename TYPE >
	type_creator type_creator::base()
	{
		m_entry->base_type = m_database->get_type( typeid( TYPE ) );
		return *this;
	}

	template< typename TOBJECT, typename TFIELD >
	type_creator type_creator::field( const char* aname, TFIELD TOBJECT::*field, typename enable_if< is_container<TFIELD>::value, void* >::type )
	{
		typedef typename TFIELD::value_type field_type;
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_entry->names.insert( name );
		f.hash = rtti_type_hash< remove_pointer_t<field_type> >::hash();
		f.type = type_db->get_type( f.hash );
		f.flags = TF_CONTAINER |
			( is_pointer<field_type>::value ? TF_POINTER : 0 ) |
			( is_pod<field_type>::value ? TF_SIMPLETYPE : 0 );
		f.offset = offset_of( field );
		m_entry->field_list.push_back( f );
		return *this;
	}

	template< typename TOBJECT, typename TFIELD >
	type_creator type_creator::field( const char* aname, TFIELD TOBJECT::*field, typename enable_if< !is_container<TFIELD>::value, void* >::type )
	{
		NV_ASSERT( m_entry->enum_list.empty(), "Type cannot have both enums and fields!" );
		type_field f;
		f.name = m_entry->names.insert( name );
		f.hash = rtti_type_hash< remove_pointer_t<TFIELD> >::hash();
		f.type = type_db->get_type( f.hash );
		f.flags =
			( is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
			( is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		f.offset = offset_of( field );
		m_entry->field_list.push_back( f );
		return *this;
	}

	inline type_creator type_creator::value( const char* name, sint32 value )
	{
		NV_ASSERT( m_entry->field_list.empty(), "Type cannot have both enums and fields!" );
		type_enum e;
		e.name = m_entry->names.insert( name );
		e.value = value;
		m_entry->enum_list.push_back( e );
		return *this;
	}

}


#endif // NV_CORE_TYPES_HH
