// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file range.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief range iterators
 */

#ifndef NV_CORE_RANGE_HH
#define NV_CORE_RANGE_HH

#include <nv/core/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/iterator.hh>
#include <nv/stl/traits/transforms.hh>

namespace nv
{

	namespace detail
	{

		template < typename T >
		class forward_iterator_base : public iterator< input_iterator_tag, T >
		{
		public:
			forward_iterator_base( T value ) : m_value( value ) {}
			T operator* () const { return m_value; }
			T const* operator-> () const { return &m_value; }
			bool operator== ( const forward_iterator_base& rhs ) const
			{
				return m_value == rhs.m_value;
			}
			bool operator!= ( const forward_iterator_base& rhs ) const
			{
				return !( *this == rhs );
			}
		protected:
			T m_value;
		};

		template < typename T >
		class range_iterator_base : public forward_iterator_base< T >
		{
		public:
			typedef forward_iterator_base< T > base_class;

			range_iterator_base( T value ) : forward_iterator_base<T>( value ) {}
			range_iterator_base& operator++ () 
			{
				++base_class::m_value; 
				return *this; 
			}
			range_iterator_base operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		};

		template < typename T >
		class range2d_iterator_base 
			: public forward_iterator_base< glm::detail::tvec2<T> >
		{
		public:
			typedef forward_iterator_base< glm::detail::tvec2<T> > base_class;

			range2d_iterator_base( glm::detail::tvec2<T> value, T min, T max ) 
				: forward_iterator_base< glm::detail::tvec2<T> >( value ), m_min(min), m_max(max) {}
			range2d_iterator_base& operator++ () 
			{
				++base_class::m_value.x;
				if ( base_class::m_value.x > m_max ) 
				{
					++base_class::m_value.y;
					base_class::m_value.x = m_min;
				}
				return *this; 
			}
			range2d_iterator_base operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		protected:
			T m_min;
			T m_max;
		};

		template < typename T >
		class bits_iterator_base : public forward_iterator_base< T >
		{
		public:
			typedef make_underlying_type_t< T > base_type;
			typedef forward_iterator_base< T > base_class;

			static const T invalid = T( 0 );

			bits_iterator_base( T value, T current ) : forward_iterator_base<T>( current ), m_full( value )
			{
				if( !( (base_type)value & (base_type)current ) )
					next();
			}
			bits_iterator_base& operator++ () 
			{
				next();
				return *this; 
			}
			bits_iterator_base operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		private:
			void next()
			{
				do
				{
					if ( base_class::m_value == invalid || m_full <= base_class::m_value ) { base_class::m_value = invalid; m_full = invalid; break; }
					base_class::m_value = T((base_type)base_class::m_value << 1);
				} while ( !( (base_type)m_full & (base_type)base_class::m_value ) );
			}

			T m_full;
		};


	} // namespace detail

	template < typename T >
	class range_iterator_provider
	{
	public:
		class iterator : public detail::range_iterator_base<T>
		{
		public:
			iterator( T value ) : detail::range_iterator_base<T>(value) {}
		};

		range_iterator_provider( T begin, T end ) : m_begin( begin ), m_end( end ) {}
		iterator begin() { return m_begin; }
		iterator end() { return m_end; }

	protected:
		iterator m_begin;
		iterator m_end;
	};

	template < typename T >
	class range2d_iterator_provider
	{
	public:
		typedef T value_type;
		typedef typename T::value_type element_type;

		class iterator : public detail::range2d_iterator_base<element_type>
		{
		public:
			iterator( value_type begin, element_type min, element_type max ) 
				: detail::range2d_iterator_base<element_type>(begin, min, max) {}
		};

		range2d_iterator_provider( value_type begin, value_type end ) : m_begin( begin, begin.x, end.x ), m_end( T(begin.x, ++end.y ), begin.x, end.x ) {}
		iterator begin() { return m_begin; }
		iterator end() { return m_end; }

	protected:
		iterator m_begin;
		iterator m_end;
	};

	template < typename T >
	class bits_iterator_provider
	{
	public:
		typedef make_underlying_type_t< T > base_type;
		typedef detail::bits_iterator_base<T> iterator;
		static const T invalid = T( 0 );

		bits_iterator_provider( T value ) : m_value( value ) {}
		iterator begin() { return iterator( m_value, T(1) ); }
		iterator end() { return iterator( m_value, invalid ); }

	protected:
		T m_value;
	};

	template < typename T >
	range_iterator_provider<T> range( T begin, T end )
	{
		return range_iterator_provider<T>( begin, end+1 );
	}

	template < typename T >
	range2d_iterator_provider<T> range2d( T begin, T end )
	{
		return range2d_iterator_provider<T>( begin, end );
	}

	template < typename T >
	range_iterator_provider<T> range( T end )
	{
		return range_iterator_provider<T>( 0, end );
	}

	template < typename T >
	range2d_iterator_provider<T> range2d( T end )
	{
		return range2d_iterator_provider<T>( T(0,0), T( --end.x, --end.y ) );
	}

	template < typename C >
	auto index( const C& c ) -> range_iterator_provider<decltype( c.size() )>
	{
		return range_iterator_provider<decltype( c.size() )>( 0, c.size() );
	}

	template < typename T >
	bits_iterator_provider<T> bits( T value )
	{
		return bits_iterator_provider<T>( value );
	}


}

#endif // NV_CORE_RANGE_HH
