// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file rtti_support.hh
 * @author Kornel Kisielewicz
 * @brief RTTI emulation
 * 
 * This is the minimal support header for our rtti to work.
 * TODO: not thread safe - once constexpr works move to runtime hash!
 */

#ifndef NV_STL_RTTI_SUPPORT_HH
#define NV_STL_RTTI_SUPPORT_HH

#define NV_RTTI_DECLARE(T) \
namespace nv { \
template<> struct rtti_type_hash< T > { \
	static const char* name() { return #T; } \
	static nv::uint64 hash() { \
NV_MSVC_SUPRESS( 4307 )\
		static const nv::uint64 value = nv::rtti_const_hash(#T); \
		return value; \
	} \
};} \

#define NV_RTTI_DECLARE_NAME(T,NAME) \
namespace nv { \
template<> struct rtti_type_hash< T > { \
	static const char* name() { return NAME; } \
	static nv::uint64 hash() { \
NV_MSVC_SUPRESS( 4307 )\
		static const nv::uint64 value = nv::rtti_const_hash(#T); \
		return value; \
				} \
};} \

namespace nv
{

	namespace detail
	{
		static constexpr unsigned long long rtti_hash_basis = 14695981039346656037ULL;
		static constexpr unsigned long long rtti_hash_prime = 1099511628211ULL;

 		constexpr uint64 rtti_hash_impl( char c, const char* remain, uint64 value )
 		{
 			return c == 0 ? value : rtti_hash_impl( remain[0], remain + 1, (uint64)(uint64)( value ^ (uint64)c ) * rtti_hash_prime );
 		}

		template < typename T >	struct rtti_type_fail { static const bool value = false; };
	}

 	// compile-time hash
 	constexpr uint64 rtti_const_hash( const char* str )
 	{
 		return detail::rtti_hash_impl( str[0], str + 1, detail::rtti_hash_basis );
 	}

	// run-time hash
	inline uint64 rtti_hash( const char* str )
	{
		uint64 hash = detail::rtti_hash_basis;
		while ( *str != 0 )
		{
			hash ^= (uint64)str[0];
			hash *= detail::rtti_hash_prime;
			++str;
		}
		return hash;
	}

	template < typename T >
	struct rtti_type_hash
	{
		static uint64 hash()
		{
			static_assert( NV_TYPE_FAIL(T), "Type not registered!" );
			return 0;
		}
		static const char* name()
		{
			static_assert( NV_TYPE_FAIL( T ), "Type not registered!" );
			return nullptr;
		}
	};

}

#endif // NV_STL_RTTI_SUPPORT_HH
