// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file memory.hh
* @author Kornel Kisielewicz
* @brief memory utilities
*/

// TODO: implement const and mutable buffer using the union trick

#ifndef NV_CORE_MEMORY_HH
#define NV_CORE_MEMORY_HH

#include <nv/core/common.hh>
#include <nv/stl/traits/properties.hh>
#include <nv/stl/traits/alignment.hh>
#include <nv/stl/utility.hh>
#include <nv/stl/iterator.hh>
#include <nv/stl/capi.hh>

namespace nv
{
	namespace detail
	{
		template< typename T >
		struct addressof_helper
		{
			T & value;
			constexpr addressof_helper( T & v ) : value( v ) {}
			constexpr operator T& () const { return value; }
		private:
			addressof_helper & operator=( const addressof_helper & );
		};

		template< typename T >
		struct addressof_impl
		{
			static constexpr T * f( T & v, long )
			{
				return reinterpret_cast<T*>(
					&const_cast<char&>( reinterpret_cast<const volatile char &>( v ) ) );
			}
			static constexpr T * f( T * v, int ) { return v; }
		};
	}

	template< typename T >
	T * addressof( T & v )
	{
		return detail::addressof_impl<T>::f( detail::addressof_helper<T>( v ), 0 );
	}

	namespace mem_flags
	{
		static constexpr uint16 is_const  = 0x0001;
		static constexpr uint16 is_static = 0x0002;
		static constexpr uint16 read_only = 0x0004;
		static constexpr uint16 temporary = 0x0008;
	}

	template< typename T >
	class storage_view
	{
	public:
		typedef T      value_type;
		typedef size_t size_type;
		static constexpr bool is_static   = false;
		static constexpr bool is_fixed    = false;
		static constexpr bool is_const    = false;
		static constexpr size_t type_size = sizeof( value_type );

		constexpr storage_view() 
			: m_data( nullptr ), m_size( 0 ) {}
		constexpr storage_view( value_type* a_data, size_type a_size )
			: m_data( a_data ), m_size( a_size ) {}

		void assign( value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size != 0; }
		constexpr const value_type* data() const { return m_data; }
		inline    value_type* data() { return m_data; }
		constexpr size_type   raw_size() const { return sizeof( value_type ) * m_size; }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }
	protected:
		value_type* m_data;
		size_type   m_size;
	};

	template< typename T >
	class const_storage_view
	{
	public:
		typedef T        value_type;
		typedef size_t   size_type;
		static constexpr bool is_static   = false;
		static constexpr bool is_fixed    = false;
		static constexpr bool is_const    = true;
		static constexpr size_t type_size = sizeof( value_type );

		constexpr const_storage_view()
			: m_data( nullptr ), m_size( 0 ) {}
		constexpr const_storage_view( const value_type* a_data, size_type a_size )
			: m_data( a_data ), m_size( a_size ) {}

		void assign( const value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		constexpr size_type size() const { return m_size; }
		constexpr bool empty() const { return m_size != 0; }
		constexpr const value_type* data() const { return m_data; }
		constexpr size_type   raw_size() const { return sizeof( value_type ) * m_size; }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
	protected:
		const value_type* m_data;
		size_type         m_size;
	};


	template < typename T >
	inline void raw_construct_object( void* object )
	{
		new ( object )T;
	}

	template < typename T, typename ...Args >
	inline void raw_construct_object( void* object, Args&&... params )
	{
		new ( object )T( forward<Args>( params )... );
	}

	template < typename T, typename ...Args >
	inline void construct_object( T* object, Args&&... params )
	{
		new ( object )T( forward<Args>( params )... );
	}

	template < typename T >
	inline void destroy_object( T* object )
	{
		object->~T();
	}

	template <typename TYPE>
	void raw_destroy_object( void* object )
	{
		( (TYPE*)object )->TYPE::~TYPE();
	}

	namespace detail
	{
		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, true_type )
		{
			fill( first, last, value );
		}

		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( (void*)&*it ) value_type( value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, true_type )
		{
			return fill_n( first, count, value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new ( (void*)&*it ) value_type( value );
			return it;
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, true_type )
		{
			fill_default( first, last );
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( (void*)&*it ) value_type;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return fill_default_n( first, count );
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new( (void*)&*it ) value_type;
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator, ForwardIterator, true_type )
		{
			// no-op
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				( *it ).~value_type();
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return first + count;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				( *it ).~value_type();
			return it;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_impl( InputIterator first, InputIterator last, ForwardIterator out, false_type )
		{
			typedef typename std::iterator_traits<ForwardIterator>::value_type value_type;
			InputIterator src( first );
			ForwardIterator dest( out );
			for ( ; first != last; ++src, ++dest )
			{
				::new ( static_cast<void*>( addressof( *dest ) ) ) value_type( *src );
			}
			return dest;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_impl( InputIterator first, InputIterator last, ForwardIterator out, true_type )
		{
			return ForwardIterator( nvmemmove( out, first, (size_t)( (uintptr_t)last - (uintptr_t)first ) ) );
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_n_impl( InputIterator first, size_t count, ForwardIterator out, false_type )
		{
			typedef typename std::iterator_traits<ForwardIterator>::value_type value_type;
			InputIterator src( first );
			ForwardIterator dest( out );
			for ( ; count > 0; --count, ++src, ++dest )
			{
				::new ( static_cast<void*>( addressof( *dest ) ) ) value_type( *src );
			}
			return it;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_n_impl( InputIterator first, size_t count, ForwardIterator out, true_type )
		{
			typedef typename std::iterator_traits<ForwardIterator>::value_type value_type;
			return ForwardIterator( nvmemmove( out, first, count * sizeof( value_type ) ) );
		}

	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill( ForwardIterator first, ForwardIterator last, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_fill_impl( first, last, value, has_trivial_assign<value_type>() );
	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill_n( ForwardIterator first, size_t count, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_fill_n_impl( first, count, value, has_trivial_assign<value_type>() );
	}

	template < typename ForwardIterator, typename ...Args >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last, Args&&... params )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		ForwardIterator it( first );
		for ( ; it != last; ++it )
			::new( (void*)&*it ) value_type( forward<Args>( params )... );
	}

	template < typename ForwardIterator >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_construct_impl( first, last, has_trivial_constructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_construct_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_construct_n_impl( first, count, has_trivial_constructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline void uninitialized_destroy( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_destroy_impl( first, last, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_destroy_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_destroy_n_impl( first, count, has_trivial_destructor<value_type>() );
	}

	template < typename InputIterator, typename ForwardIterator >
	inline ForwardIterator uninitialized_copy( InputIterator first, InputIterator last, ForwardIterator out )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_copy_impl( first, last, out, has_trivial_copy<value_type>() );
	}

	template < typename InputIterator, typename ForwardIterator >
	inline ForwardIterator uninitialized_copy_n( InputIterator first, size_t count, ForwardIterator out )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_copy_n_impl( first, count, out, has_trivial_copy<value_type>() );
	}

	namespace detail
	{

		template < typename Super, bool Const = Super::is_const >
		class add_iterators {};

		template < typename Super >
		class add_iterators < Super, true > : public Super
		{
		public:
			typedef typename Super::value_type           value_type;
			typedef const value_type*                    const_iterator;
			typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

			using Super::Super;

			inline const_iterator begin() const { return const_iterator( Super::data() ); }
			inline const_iterator end() const { return const_iterator( Super::data() + Super::size() ); }
			inline const_iterator cbegin() const { return const_iterator( Super::data() ); }
			inline const_iterator cend() const { return const_iterator( Super::data() + Super::size() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
		};

		template < typename Super >
		class add_iterators < Super, false > : public Super
		{
		public:
			typedef typename Super::value_type           value_type;
			typedef value_type*                          iterator;
			typedef const value_type*                    const_iterator;
			typedef nv::reverse_iterator<iterator>       reverse_iterator;
			typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

			using Super::Super;

			inline const_iterator begin() const { return const_iterator( Super::data() ); }
			inline const_iterator end() const { return const_iterator( Super::data() + Super::size() ); }
			inline iterator begin() { return iterator( Super::data() ); }
			inline iterator end() { return iterator( Super::data() + Super::size() ); }
			inline const_iterator cbegin() const { return const_iterator( Super::data() ); }
			inline const_iterator cend() const { return const_iterator( Super::data() + Super::size() ); }
			inline reverse_iterator rbegin() { return reverse_iterator( end() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline reverse_iterator rend() { return reverse_iterator( begin() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
			inline iterator       iat( size_t i ) { NV_ASSERT( i <= Super::size(), "Index out of range" ); return begin() + i; }
		};

		template < typename Super, bool Const = Super::is_const >
		class add_random_access {};

		template < typename Super >
		class add_random_access < Super, true > : public Super
		{
		public:
			typedef typename Super::value_type           value_type;
			typedef typename Super::size_type            size_type;
			typedef const value_type*                    const_iterator;
			typedef value_type&                          reference;
			typedef const value_type&                    const_reference;

			inline const_reference front() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[0]; }
			inline const_reference back() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }

			const_reference operator[]( size_type i ) const
			{
				NV_ASSERT( i < Super::size(), "Out of range" );
				return Super::data()[i];
			}
		};

		template < typename Super >
		class add_random_access < Super, false > : public Super
		{
		public:
			typedef typename Super::value_type           value_type;
			typedef typename Super::size_type            size_type;
			typedef value_type*                          iterator;
			typedef const value_type*                    const_iterator;
			typedef value_type&                          reference;
			typedef const value_type&                    const_reference;

			inline reference       front() { NV_ASSERT( !Super::empty(), "front() called on empty data!" );  return Super::data()[0]; }
			inline const_reference front() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[0]; }
			inline reference       back() { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }
			inline const_reference back() const { NV_ASSERT( !Super::empty(), "front() called on empty data!" ); return Super::data()[Super::size() - 1]; }

			reference operator[]( size_type i )
			{
				NV_ASSERT( i < Super::size(), "Out of range" );
				return Super::data()[i];
			}

			const_reference operator[]( size_type i ) const
			{
				NV_ASSERT( i < Super::size(), "Out of range" );
				return Super::data()[i];
			}

			inline void fill( const value_type& value )
			{
				fill_n( iterator( Super::data() ), iterator( Super::data() + this->size() ), value );
			}

		};

	}

	class const_mem_ref : public detail::add_iterators< const_storage_view< char > >
	{
		typedef detail::add_iterators< const_storage_view< char > > inherited;
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef const_pointer        const_iterator;
	public:
		inline const_mem_ref() : inherited() {}
		inline const_mem_ref( const void* p, size_type n ) : inherited( (const char*)p, n ) {}
		inline const_mem_ref( const const_mem_ref& l ) : inherited( l.data(), l.size() ) {}
	};

	class mem_ref : public detail::add_iterators< storage_view< char > >
	{
		typedef detail::add_iterators< storage_view< char > > inherited;
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef pointer              iterator;
		typedef const_pointer        const_iterator;
	public:
		inline mem_ref() : inherited() {}
		inline mem_ref( void* p, size_type n ) : inherited( (char*)p, n ) {}
		inline mem_ref( const mem_ref& l ) : inherited( const_cast< char* >( l.data() ), l.size() ) {}
	};

	template< typename SizeType >
	struct default_next_capacity
	{
		static SizeType get( SizeType requested, SizeType capacity )
		{
			SizeType additional = nv::max( requested, capacity );
			return capacity + additional;
		}
	};

	struct policy_initialize_always
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator first, ForwardIterator last )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			uninitialized_construct( first, last, value_type() );
		}
		template < typename InputIterator, typename ForwardIterator >
		static ForwardIterator copy( InputIterator first, InputIterator last, ForwardIterator out )
		{
			return uninitialized_copy( first, last, out );
		}
		template < typename ForwardIterator >
		static void destroy( ForwardIterator first, ForwardIterator last )
		{
			uninitialized_destroy( first, last );
		}
	};

	struct policy_initialize_never
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator, ForwardIterator )
		{
		}
		template < typename InputIterator, typename ForwardIterator >
		static ForwardIterator copy( InputIterator first, InputIterator last, ForwardIterator out )
		{
			return detail::uninitialized_copy( first, last, out, true_type );
		}
		template < typename ForwardIterator >
		static void destroy( ForwardIterator, ForwardIterator )
		{
		}
	};

	struct policy_initialize_standard
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator first, ForwardIterator last )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			if ( !has_trivial_constructor<value_type>() )
				detail::uninitialized_construct_impl( first, last, false_type() );
		}
		template < typename InputIterator, typename ForwardIterator >
		static ForwardIterator copy( InputIterator first, InputIterator last, ForwardIterator out )
		{
			return uninitialized_copy( first, last, out );
		}
		template < typename ForwardIterator >
		static void destroy( ForwardIterator first, ForwardIterator last )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			if ( !has_trivial_destructor<value_type>() )
				detail::uninitialized_destroy_impl( first, last, false_type() );
		}
	};

	template< typename T >
	class dynamic_storage
	{
	public:
		typedef T      value_type;

		static constexpr bool is_static   = false;
		static constexpr bool is_const    = false;
		static constexpr size_t type_size = sizeof( value_type );

		constexpr dynamic_storage() : m_data( nullptr ) {}

		constexpr const value_type* data() const { return reinterpret_cast<const value_type*>( m_data ); }
		inline    value_type* data() { return reinterpret_cast<T*>( m_data ); }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }
	protected:
		bool reallocate( size_t new_size )
		{
			if ( m_data != 0 || new_size != 0 )
				m_data = (uint8*)nvrealloc( m_data, new_size * sizeof( value_type ) );
			return true; // TODO : alloc check?
		}
	protected:
		uint8* m_data;
	};

	template< typename T, size_t N >
	class static_storage
	{
	public:
		typedef T      value_type;

		static constexpr bool is_static   = true;
		static constexpr bool is_const    = false;
		static constexpr size_t type_size = sizeof( value_type );

		constexpr const value_type* data() const { return reinterpret_cast<const value_type*>( m_data ); }
		inline    value_type* data() { return reinterpret_cast<T*>( m_data ); }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }
	protected:
		static constexpr bool reallocate( size_t new_size ) { return new_size <= N; }
	protected:
		typedef aligned_array_t<T, N, alignof( T ) > storage_type;
		storage_type m_data;
	};

	template< typename Storage, size_t N >
	class fixed_storage : public Storage
	{
	public:
		typedef size_t size_type;

		static constexpr bool is_fixed = true;

		fixed_storage()
		{
			Storage::reallocate( N );
		}
		~fixed_storage()
		{
			Storage::reallocate( 0 );
		}
		static constexpr size_t capacity() { return N; }
		static constexpr size_t size() { return N; }
		static constexpr bool empty() { return N != 0; }
		static constexpr size_t raw_size() { return sizeof( T ) * N; }
	};

	template< typename Storage >
	class resizable_storage : public Storage
	{
	public:
		typedef size_t size_type;

		static constexpr bool is_fixed = false;

		~resizable_storage()
		{
			if ( m_size > 0 ) reallocate( 0 );
		}
		constexpr size_t capacity() { return m_size; }
		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size != 0; }
		constexpr size_t raw_size() const { return sizeof( value_type ) * m_size; }
	protected:
		constexpr resizable_storage() : m_size( 0 ) {}
		// TODO: return type error checking
		bool resize( size_t new_size )
		{
			if ( new_size != m_size )
			{
				m_size = new_size;
				return reallocate( m_size );
			}
			return true;
		}
	protected:
		size_type m_size;
	};


	template< typename T, size_t N >
	using fixed_static_storage  = fixed_storage< static_storage< T, N >, N >;

	template< typename T, size_t N >
	using fixed_dynamic_storage = fixed_storage< dynamic_storage< T >, N >;

	template< typename T >
	using resizable_dynamic_storage = resizable_storage< dynamic_storage< T > >;

// TODO:
//	template< typename T, size_t N >
//	using resizable_static_storage  = resizable_storage< static_storage< T, N > >;

// TODO:
//	template< typename Storage, typename NextCapacity = default_next_capacity >
//	class growable_storage : Storage;

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class fixed_container_allocator : public Storage
	{
	public:
		typedef typename Storage::value_type value_type;
		typedef typename Storage::size_type  size_type;

		fixed_container_allocator()
		{
			InitializePolicy::initialize( data(), data() + Storage::capacity() );
		}

		~fixed_container_allocator()
		{
			InitializePolicy::destroy( data(), data() + Storage::capacity() );
		}
	};

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class sized_container_allocator : public Storage
	{
	public:
		typedef typename Storage::value_type value_type;
		typedef typename Storage::size_type  size_type;

		sized_container_allocator() {}
		explicit sized_container_allocator( size_type new_size ) { resize( new_size ); }

		void resize( size_type new_size )
		{
			size_type old_size = Storage::size();
			if ( new_size != old_size )
			{
				if ( new_size < old_size )
				{
					InitializePolicy::destroy( Storage::data() + new_size, Storage::data() + old_size );
				}
				Storage::resize( new_size );
				if ( new_size > old_size )
				{
					InitializePolicy::initialize( Storage::data() + old_size, Storage::data() + new_size );
				}
			}
		}

		void assign( const value_type* ptr, size_type sz )
		{
			if ( Storage::size() > 0 ) InitializePolicy::destroy( Storage::data(), Storage::data() + Storage::size() );
			if ( ptr != nullptr && sz > 0 )
			{
				if ( sz != Storage::size() )
				{
					Storage::resize( 0 );
					Storage::resize( sz );
				}
				InitializePolicy::copy( ptr, ptr + sz, Storage::data() );
			}
			else Storage::resize( 0 );
		}

		void clear()
		{
			if ( Storage::size() > 0 )
			{
				InitializePolicy::destroy( Storage::data(), Storage::data() + Storage::size() );
				Storage::resize( 0 );
			}
		}

		~sized_container_allocator()
		{
			if ( Storage::size() > 0 ) clear();
		}

	};

}

#endif // NV_CORE_MEMORY_HH
