// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
* @file transforms.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief type traits - transforms
*/

#ifndef NV_STL_TRAITS_TRANSFORMS_HH
#define NV_STL_TRAITS_TRANSFORMS_HH

#include <nv/core/common.hh>
#include <nv/stl/traits/common.hh>
#include <nv/stl/traits/primary.hh>

namespace nv
{

	namespace detail
	{

		template< typename T, bool CONST, bool VOLATILE>
		struct cv_selector;

		template< typename T >
		struct cv_selector < T, false, false > { typedef T type; };

		template< typename T >
		struct cv_selector < T, true, false > { typedef const T type; };

		template< typename T >
		struct cv_selector < T, false, true > { typedef volatile T type; };

		template< typename T >
		struct cv_selector < T, true, true > { typedef const volatile T type; };

		template< typename SOURCE, typename TARGET,
			bool CONST = is_const<SOURCE>::value,
			bool VOLATILE = is_volatile<SOURCE>::value >
		struct match_cv
		{
			typedef typename cv_selector< TARGET, CONST, VOLATILE >::type type;
		};

		template < typename T >
		struct signed_type
		{
			typedef T type;
		};

		template<> struct signed_type < char > { typedef signed char type; };
		template<> struct signed_type < unsigned char > { typedef signed char type; };
		template<> struct signed_type < unsigned short > { typedef signed short type; };
		template<> struct signed_type < unsigned int > { typedef signed int type; };
		template<> struct signed_type < unsigned long > { typedef signed long type; };
		template<> struct signed_type < unsigned long long > { typedef signed long long type; };

		template < typename T >
		struct unsigned_type
		{
			typedef T type;
		};

		template<> struct unsigned_type < char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed short > { typedef unsigned short type; };
		template<> struct unsigned_type < signed int > { typedef unsigned int type; };
		template<> struct unsigned_type < signed long > { typedef unsigned long type; };
		template<> struct unsigned_type < signed long long > { typedef unsigned long long type; };

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_signed_impl;

		template < typename T >
		struct make_signed_impl < T, false >
		{
		private:
			typedef signed_type< remove_cv_t<T> > signed_type_result;
		public:
			typedef match_cv< T, typename signed_type_result::type > type;
		};

		template < typename T >
		struct make_signed_impl < T, true >
		{
		private:
			static constexpr bool size1test = sizeof( T ) <= sizeof( signed char );
			static constexpr bool size2test = sizeof( T ) <= sizeof( signed short );
			static constexpr bool size4test = sizeof( T ) <= sizeof( signed int );
			typedef conditional_t<size4test, signed int, signed long> test4type;
			typedef conditional_t<size2test, signed short, test4type> test2type;
		public:
			typedef conditional_t<size1test, signed char, test2type> type;
		};

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_unsigned_impl;

		template < typename T >
		struct make_unsigned_impl < T, false >
		{
		private:
			typedef unsigned_type<remove_cv_t<T>> unsigned_type_result;
		public:
			typedef match_cv< T, typename unsigned_type_result::type > type;
		};

		template < typename T >
		struct make_unsigned_impl < T, true >
		{
		private:
			static constexpr bool size1test = sizeof( T ) <= sizeof( unsigned char );
			static constexpr bool size2test = sizeof( T ) <= sizeof( unsigned short );
			static constexpr bool size4test = sizeof( T ) <= sizeof( unsigned int );
			typedef conditional_t<size4test, unsigned int, unsigned long> test4type;
			typedef conditional_t<size2test, unsigned short, test4type> test2type;
		public:
			typedef conditional_t<size1test, unsigned char, test2type> type;
		};

	}

	template < typename T >
	struct make_signed
	{
		typedef typename detail::make_signed_impl<T>::type type;
	};

	template <> struct make_signed < bool > ;

	template < typename T >
	using make_signed_t = typename make_signed<T>::type;

	template < typename T >
	struct make_unsigned
	{
		typedef typename detail::make_unsigned_impl<T>::type type;
	};

	template <> struct make_unsigned < bool > ;

	template < typename T >
	using make_unsigned_t = typename make_unsigned<T>::type;


#if NV_COMPILER == NV_MSVC || NV_COMPILER == NV_CLANG
	template < typename T >
	struct underlying_type
	{
		typedef __underlying_type( T ) type;
	};
#else
	template< typename T >
	struct underlying_type
	{
		typedef typename conditional <
			T( -1 ) < T( 0 ),
			typename make_signed< T >::type,
			typename make_unsigned< T >::type
			> ::type type;
	};
#endif
	template < typename T >
	using underlying_type_t = typename underlying_type<T>::type;

	namespace detail
	{

		template< typename T, bool IsEnum >
		struct make_underlying_type_impl
		{
			typedef T type;
		};

		template< typename T >
		struct make_underlying_type_impl< T, true >
		{
			typedef typename underlying_type<T>::type type;
		};

	}

	template < typename T >
	struct make_underlying_type
	{
		typedef typename detail::make_underlying_type_impl< T, is_enum<T>::value >::type type;
	};

	template < typename T >
	using make_underlying_type_t = typename make_underlying_type<T>::type;

	template< typename T >
	struct decay
	{
		typedef remove_reference_t<T> U;
		typedef conditional_t <
			is_array<U>::value,
			remove_extent_t<U>*,
			conditional_t <
				is_function<U>::value,
				add_pointer_t<U>,
				remove_cv_t<U>
			>
		> type;
	};

	template< typename T > 
	using decay_t = typename decay<T>::type;

	template <class ...T> struct common_type;

	template < typename T >
	struct common_type < T >
	{
		typedef typename decay<T>::type type;
	};

	template < typename T1, typename T2 >
	struct common_type < T1, T2 >
	{
		typedef typename decay<
			decltype( true ? declval<T1>() : declval<T2>()
			)>::type type;
	};

	template < typename T1, typename T2, typename... Types >
	struct common_type < T1, T2, Types... >
	{
		typedef typename common_type<
			typename common_type<T1, T2>::type,
			Types...>::type type;
	};

	template < typename... Types >
	using common_type_t = typename common_type<Types...>::type;

	namespace detail
	{

		template < typename F, typename... Args >
		inline auto result_call( F&& f, Args&&... args )
			-> decltype( forward<F>( f )( forward<Args>( args )... ) )
		{
			return forward<F>( f )( forward<Args>( args )... );
		}

		template < typename Base, typename T, typename Derived >
		inline auto result_call( T Base::*pmd, Derived&& ref )
			-> decltype( forward<Derived>( ref ).*pmd )
		{
			return forward<Derived>( ref ).*pmd;
		}

		template < typename PMD, typename Pointer >
		inline auto result_call( PMD&& pmd, Pointer&& ptr )
			-> decltype( ( *forward<Pointer>( ptr ) ).*forward<PMD>( pmd ) )
		{
			return ( *forward<Pointer>( ptr ) ).*forward<PMD>( pmd );
		}

		template < typename Base, typename T, typename Derived, typename... Args >
		inline auto result_call( T Base::*pmf, Derived&& ref, Args&&... args )
			-> decltype( ( forward<Derived>( ref ).*pmf )( forward<Args>( args )... ) )
		{
			return ( forward<Derived>( ref ).*pmf )( forward<Args>( args )... );
		}

		template < typename PMF, typename Pointer, typename... Args >
		inline auto result_call( PMF&& pmf, Pointer&& ptr, Args&&... args )
			-> decltype( ( ( *forward<Pointer>( ptr ) ).*forward<PMF>( pmf ) )( forward<Args>( args )... ) )
		{
			return ( ( *forward<Pointer>( ptr ) ).*forward<PMF>( pmf ) )( forward<Args>( args )... );
		}

		// TODO: fix this once compiler takes it
		// 		template < typename, typename = void >
		// 		struct result_of_impl {};
		// 
		// 		template < typename F, typename...Args >
		// 		struct result_of_impl < F( Args... ),
		// 			decltype( void( result_call( declval_i<F>(), declval_i<Args>()... ) ) ) >
		// 		{
		// 			using type = decltype( result_call( declval_i<F>(), declval_i<Args>()... ) );
		// 		};

		template < typename > struct result_of_impl;
		template < typename F, typename... Args >
		struct result_of_impl < F( Args... ) >
		{
			using type = decltype( result_call( declval<F>(), declval<Args>()... ) );
		};
	}

	template < typename T >
	struct result_of : detail::result_of_impl < T > {};

	template < typename T >
	using result_of_t = typename result_of<T>::type;
}

#endif // NV_STL_TRAITS_TRANSFORMS_HH
