// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file limits.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief STL limits library
*/

#ifndef NV_STL_LIMITS_HH
#define NV_STL_LIMITS_HH

#include <nv/core/common.hh>

namespace nv
{
	struct limits
	{
		static constexpr unsigned char uc_min = 0;
		static constexpr unsigned char uc_max = 255;
		static constexpr unsigned char uc_bit = sizeof( unsigned char ) * 8;
		static constexpr signed char sc_min = -128;
		static constexpr signed char sc_max = 127;
		static constexpr signed char sc_bit = sizeof( signed char ) * 8;
		static constexpr char c_min = ( char( 0 ) < char( -1 ) ? 0 : -128 );
		static constexpr char c_max = ( char( 0 ) < char( -1 ) ? 255 : 127 );
		static constexpr char c_bit = sizeof( char ) * 8;
		static constexpr unsigned short us_min = 0;
		static constexpr unsigned short us_max = 65535;
		static constexpr unsigned short us_bit = sizeof( unsigned short ) * 8;
		static constexpr signed short ss_min = -32768;
		static constexpr signed short ss_max = 32767;
		static constexpr signed short ss_bit = sizeof( signed short ) * 8;
		static constexpr unsigned int ui_min = 0;
		static constexpr unsigned int ui_max = 4294967295;
		static constexpr unsigned int ui_bit = sizeof( unsigned int ) * 8;
		static constexpr signed int si_min = -2147483648;
		static constexpr signed int si_max = 2147483647;
		static constexpr signed int si_bit = sizeof( signed int ) * 8;
		static constexpr unsigned long ul_min = 0;
		static constexpr unsigned long ul_max = 4294967295;
		static constexpr unsigned long ul_bit = sizeof( unsigned long ) * 8;
		static constexpr signed long sl_min = -2147483648;
		static constexpr signed long sl_max = 2147483647;
		static constexpr signed long sl_bit = sizeof( signed long ) * 8;
		static constexpr unsigned long long ull_min = 0;
		static constexpr unsigned long long ull_max = 18446744073709551615;
		static constexpr unsigned long long ull_bit = sizeof( unsigned long long ) * 8;
		static constexpr signed long long sll_min = -9223372036854775808;
		static constexpr signed long long sll_max = 9223372036854775807;
		static constexpr signed long long sll_bit = sizeof( signed long long ) * 8;

		static constexpr int f_dig        = 6;
		static constexpr int f_mant_dig   = 24;
		static constexpr int f_max_exp    = 128;
		static constexpr int f_max_10_exp = 38;
		static constexpr int f_min_exp    = -125;
		static constexpr int f_min_10_exp = -37;

		static constexpr int d_dig        = 15;
		static constexpr int d_mant_dig   = 53;
		static constexpr int d_max_exp    = 1024;
		static constexpr int d_max_10_exp = 308;
		static constexpr int d_min_exp    = -1021;
		static constexpr int d_min_10_exp = -307;

		static constexpr int ld_dig        = d_dig;
		static constexpr int ld_mant_dig   = d_mant_dig;
		static constexpr int ld_max_exp    = d_max_exp;
		static constexpr int ld_max_10_exp = d_max_10_exp;
		static constexpr int ld_min_exp    = d_min_exp;
		static constexpr int ld_min_10_exp = d_min_10_exp;
	};

	enum float_denorm_style
	{
		denorm_indeterminate = -1,
		denorm_absent        = 0,
		denorm_present       = 1
	};

	enum float_round_style
	{
		round_indeterminate       = -1,
		round_toward_zero         = 0,
		round_to_nearest          = 1,
		round_toward_infinity     = 2,
		round_toward_neg_infinity = 3
	};

	namespace detail
	{
		struct numeric_limits_base
		{
			static constexpr float_denorm_style has_denorm = denorm_absent;
			static constexpr bool has_denorm_loss   = false;
			static constexpr bool has_infinity      = false;
			static constexpr bool has_quiet_NaN     = false;
			static constexpr bool has_signaling_NaN = false;
			static constexpr bool is_bounded        = false;
			static constexpr bool is_exact          = false;
			static constexpr bool is_iec559         = false;
			static constexpr bool is_integer        = false;
			static constexpr bool is_modulo         = false;
			static constexpr bool is_signed         = false;
			static constexpr bool is_specialized    = false;
			static constexpr bool tinyness_before   = false;
			static constexpr bool traps             = false;
			static constexpr float_round_style round_style = round_toward_zero;
			static constexpr int digits         = 0;
			static constexpr int digits10       = 0;
			static constexpr int max_digits10   = 0;
			static constexpr int max_exponent   = 0;
			static constexpr int max_exponent10 = 0;
			static constexpr int min_exponent   = 0;
			static constexpr int min_exponent10 = 0;
			static constexpr int radix          = 0;
		};

		struct numeric_limits_int_base : public numeric_limits_base
		{
			static constexpr bool is_bounded     = true;
			static constexpr bool is_exact       = true;
			static constexpr bool is_integer     = true;
			static constexpr bool is_modulo      = true;
			static constexpr bool is_specialized = true;
			static constexpr int radix = 2;
		};

		struct numeric_limits_float_base : public numeric_limits_base
		{
			static constexpr float_denorm_style has_denorm = denorm_present;
			static constexpr bool has_denorm_loss   = true;
			static constexpr bool has_infinity      = true;
			static constexpr bool has_quiet_NaN     = true;
			static constexpr bool has_signaling_NaN = true;
			static constexpr bool is_bounded        = true;
			static constexpr bool is_iec559         = true;
			static constexpr bool is_signed         = true;
			static constexpr bool is_specialized    = true;
			static constexpr bool tinyness_before   = true;
			static constexpr float_round_style round_style = round_to_nearest;
			static constexpr int radix = 2;
		};

	}

	template< typename T >
	struct numeric_limits : detail::numeric_limits_base
	{
		static constexpr T min()           noexcept { return T( 0 ); }
		static constexpr T max()           noexcept { return T( 0 ); }
		static constexpr T lowest()        noexcept { return T( 0 ); }
		static constexpr T epsilon()       noexcept { return T( 0 ); }
		static constexpr T round_error()   noexcept { return T( 0 ); }
	};

	template< typename T >
	struct numeric_limits< const T > : numeric_limits < T > {};

	template< typename T >
	struct numeric_limits< volatile T > : numeric_limits < T > {};

	template< typename T >
	struct numeric_limits< const volatile T > : numeric_limits < T > {};

	template<>
	struct numeric_limits < bool > : detail::numeric_limits_int_base
	{
		static constexpr bool min()           noexcept { return false; }
		static constexpr bool max()           noexcept { return true; }
		static constexpr bool lowest()        noexcept { return false; }
		static constexpr bool epsilon()       noexcept { return bool( 0 ); }
		static constexpr bool round_error()   noexcept { return bool( 0 ); }

		static constexpr bool is_modulo = false;
		static constexpr bool is_signed = false;
		static constexpr int  digits    = 1;
		static constexpr int  digits10  = 0;
	};

	template<>
	struct numeric_limits < char > : detail::numeric_limits_int_base
	{
		static constexpr char min()           noexcept { return limits::c_min; }
		static constexpr char max()           noexcept { return limits::c_max; }
		static constexpr char lowest()        noexcept { return limits::c_min; }
		static constexpr char epsilon()       noexcept { return 0; }
		static constexpr char round_error()   noexcept { return 0; }

		static constexpr bool is_signed = (limits::c_min < 0);
		static constexpr int  digits    = limits::c_bit - ( limits::c_min < 0 ? 1 : 0 );
		static constexpr int  digits10  = (limits::c_bit - ( limits::c_min < 0 ? 1 : 0 )) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned char > : detail::numeric_limits_int_base
	{
		static constexpr unsigned char min()           noexcept { return limits::uc_min; }
		static constexpr unsigned char max()           noexcept { return limits::uc_max; }
		static constexpr unsigned char lowest()        noexcept { return limits::uc_min; }
		static constexpr unsigned char epsilon()       noexcept { return 0; }
		static constexpr unsigned char round_error()   noexcept { return 0; }

		static constexpr bool is_signed = false;
		static constexpr int  digits    = limits::c_bit;
		static constexpr int  digits10  = limits::c_bit  * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed char > : detail::numeric_limits_int_base
	{
		static constexpr signed char min()           noexcept { return limits::sc_min; }
		static constexpr signed char max()           noexcept { return limits::sc_max; }
		static constexpr signed char lowest()        noexcept { return limits::sc_min; }
		static constexpr signed char epsilon()       noexcept { return 0; }
		static constexpr signed char round_error()   noexcept { return 0; }

		static constexpr bool is_signed = true;
		static constexpr int  digits    = limits::c_bit - 1;
		static constexpr int  digits10  = (limits::c_bit - 1) * 301 / 1000;
	};


	template<>
	struct numeric_limits < unsigned short > : detail::numeric_limits_int_base
	{
		static constexpr unsigned short min()           noexcept { return limits::us_min; }
		static constexpr unsigned short max()           noexcept { return limits::us_max; }
		static constexpr unsigned short lowest()        noexcept { return limits::us_min; }
		static constexpr unsigned short epsilon()       noexcept { return 0; }
		static constexpr unsigned short round_error()   noexcept { return 0; }

		static constexpr bool is_signed = false;
		static constexpr int  digits    = limits::us_bit;
		static constexpr int  digits10  = limits::us_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed short > : detail::numeric_limits_int_base
	{
		static constexpr signed short min()           noexcept { return limits::ss_min; }
		static constexpr signed short max()           noexcept { return limits::ss_max; }
		static constexpr signed short lowest()        noexcept { return limits::ss_min; }
		static constexpr signed short epsilon()       noexcept { return 0; }
		static constexpr signed short round_error()   noexcept { return 0; }

		static constexpr bool is_signed = true;
		static constexpr int  digits    = limits::ss_bit - 1;
		static constexpr int  digits10  = ( limits::ss_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned int > : detail::numeric_limits_int_base
	{
		static constexpr unsigned int min()           noexcept { return limits::ui_min; }
		static constexpr unsigned int max()           noexcept { return limits::ui_max; }
		static constexpr unsigned int lowest()        noexcept { return limits::ui_min; }
		static constexpr unsigned int epsilon()       noexcept { return 0; }
		static constexpr unsigned int round_error()   noexcept { return 0; }

		static constexpr bool is_signed = false;
		static constexpr int  digits    = limits::ui_bit;
		static constexpr int  digits10  = limits::ui_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed int > : detail::numeric_limits_int_base
	{
		static constexpr signed int min()           noexcept { return limits::si_min; }
		static constexpr signed int max()           noexcept { return limits::si_max; }
		static constexpr signed int lowest()        noexcept { return limits::si_min; }
		static constexpr signed int epsilon()       noexcept { return 0; }
		static constexpr signed int round_error()   noexcept { return 0; }

		static constexpr bool is_signed = true;
		static constexpr int  digits    = limits::si_bit - 1;
		static constexpr int  digits10  = ( limits::si_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned long > : detail::numeric_limits_int_base
	{
		static constexpr unsigned long min()           noexcept { return limits::ul_min; }
		static constexpr unsigned long max()           noexcept { return limits::ul_max; }
		static constexpr unsigned long lowest()        noexcept { return limits::ul_min; }
		static constexpr unsigned long epsilon()       noexcept { return 0; }
		static constexpr unsigned long round_error()   noexcept { return 0; }

		static constexpr bool is_signed = false;
		static constexpr int  digits    = limits::ul_bit;
		static constexpr int  digits10  = limits::ul_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed long > : detail::numeric_limits_int_base
	{
		static constexpr signed long min()           noexcept { return limits::sl_min; }
		static constexpr signed long max()           noexcept { return limits::sl_max; }
		static constexpr signed long lowest()        noexcept { return limits::sl_min; }
		static constexpr signed long epsilon()       noexcept { return 0; }
		static constexpr signed long round_error()   noexcept { return 0; }

		static constexpr bool is_signed = true;
		static constexpr int  digits    = limits::sl_bit - 1;
		static constexpr int  digits10  = ( limits::sl_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned long long > : detail::numeric_limits_int_base
	{
		static constexpr unsigned long long min()           noexcept { return limits::ull_min; }
		static constexpr unsigned long long max()           noexcept { return limits::ull_max; }
		static constexpr unsigned long long lowest()        noexcept { return limits::ull_min; }
		static constexpr unsigned long long epsilon()       noexcept { return 0; }
		static constexpr unsigned long long round_error()   noexcept { return 0; }

		static constexpr bool is_signed = false;
		static constexpr int  digits    = limits::ull_bit;
		static constexpr int  digits10  = limits::ull_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed long long > : detail::numeric_limits_int_base
	{
		static constexpr signed long long min()           noexcept { return limits::sll_min; }
		static constexpr signed long long max()           noexcept { return limits::sll_max; }
		static constexpr signed long long lowest()        noexcept { return limits::sll_min; }
		static constexpr signed long long epsilon()       noexcept { return 0; }
		static constexpr signed long long round_error()   noexcept { return 0; }

		static constexpr bool is_signed = true;
		static constexpr int  digits    = limits::sll_bit - 1;
		static constexpr int  digits10  = ( limits::sll_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < float > : detail::numeric_limits_float_base
	{
		static constexpr float min()           noexcept { return 1.175494351e-38F; }
		static constexpr float max()           noexcept { return 3.402823466e+38F; }
		static constexpr float lowest()        noexcept { return -max(); }
		static constexpr float epsilon()       noexcept { return 1.192092896e-07F; }
		static constexpr float round_error()   noexcept { return 0.5F; }

		static constexpr int digits         = limits::f_mant_dig;
		static constexpr int digits10       = limits::f_dig;
		static constexpr int max_digits10   = 2 + limits::f_dig * 301 / 1000;
		static constexpr int max_exponent   = limits::f_max_exp;
		static constexpr int max_exponent10 = limits::f_max_10_exp;
		static constexpr int min_exponent   = limits::f_min_exp;
		static constexpr int min_exponent10 = limits::f_min_10_exp;
	};

	template<>
	struct numeric_limits < double > : detail::numeric_limits_float_base
	{
		static constexpr double min()           noexcept { return 2.2250738585072014e-308; }
		static constexpr double max()           noexcept { return 1.7976931348623158e+308; }
		static constexpr double lowest()        noexcept { return -max(); }
		static constexpr double epsilon()       noexcept { return 2.2204460492503131e-016; }
		static constexpr double round_error()   noexcept { return 0.5; }

		static constexpr int digits         = limits::d_mant_dig;
		static constexpr int digits10       = limits::d_dig;
		static constexpr int max_digits10   = 2 + limits::d_dig * 301 / 1000;
		static constexpr int max_exponent   = limits::d_max_exp;
		static constexpr int max_exponent10 = limits::d_max_10_exp;
		static constexpr int min_exponent   = limits::d_min_exp;
		static constexpr int min_exponent10 = limits::d_min_10_exp;
	};

	template<>
	struct numeric_limits < long double > : detail::numeric_limits_float_base
	{
		static constexpr long double min()           noexcept { return 2.2250738585072014e-308; }
		static constexpr long double max()           noexcept { return 1.7976931348623158e+308; }
		static constexpr long double lowest()        noexcept { return -max(); }
		static constexpr long double epsilon()       noexcept { return 2.2204460492503131e-016; }
		static constexpr long double round_error()   noexcept { return 0.5; }

		static constexpr int digits         = limits::ld_mant_dig;
		static constexpr int digits10       = limits::ld_dig;
		static constexpr int max_digits10   = 2 + limits::ld_dig * 301 / 1000;
		static constexpr int max_exponent   = limits::ld_max_exp;
		static constexpr int max_exponent10 = limits::ld_max_10_exp;
		static constexpr int min_exponent   = limits::ld_min_exp;
		static constexpr int min_exponent10 = limits::ld_min_10_exp;
	};

}

#endif // NV_STL_LIMITS_HH