// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file type_traits.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type traits
 */
// TODO: "......" function match version?
// TODO: remove typeinfo?


#ifndef NV_STL_TYPE_TRAITS_HH
#define NV_STL_TYPE_TRAITS_HH

#include <nv/core/common.hh>

namespace nv
{

	template < typename T1, typename T2 >
	struct is_same : false_type {};

	template < typename T >
	struct is_same < T, T > : true_type {};

	template < typename T >
	struct is_array : false_type {};

	template < typename T, size_t N >
	struct is_array < T[N] > : true_type {};

	template < typename T >
	struct is_array < T[] > : true_type {};

	template < typename T >
	struct is_lvalue_reference : false_type {};

	template < typename T >
	struct is_lvalue_reference < T& > : true_type{};

	template < typename T >
	struct is_rvalue_reference : false_type {};

	template < typename T >
	struct is_rvalue_reference < T&& > : true_type{};

	template < typename _Ty>
	struct is_reference : integral_constant < bool, is_lvalue_reference<_Ty>::value || is_rvalue_reference<_Ty>::value > {};

	template < typename T >
	struct is_enum : integral_constant < bool, __is_enum( T ) > {};

	// TODO: these seem to simple compared to MSVC/GCC - so we'll leave them in
	// detail - research why it is so.
	template < typename T >
	struct is_const : false_type {};
	template < typename T >
	struct is_const < T const > : true_type{};
	template < typename T >
	struct is_volatile : false_type {};
	template < typename T >
	struct is_volatile < T volatile > : true_type{};
	// TODO END

	template< typename T >
	struct remove_reference
	{
		typedef T type;
	};

	template< typename T >
	struct remove_reference < T& >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_reference < T&& >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const < const T >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const < const T[] >
	{
		typedef T type[];
	};

	template< typename T, unsigned int N >
	struct remove_const < const T[N] >
	{
		typedef T type[N];
	};

	template< typename T >
	struct remove_volatile
	{
		typedef T type;
	};

	template< typename T >
	struct remove_volatile < volatile T >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_volatile < volatile T[] >
	{
		typedef T type[];
	};

	template< typename T, unsigned int N >
	struct remove_volatile < volatile T[N] >
	{
		typedef T type[N];
	};

	template< typename T >
	struct remove_cv
	{
		typedef typename remove_const< typename remove_volatile<T>::type >::type
			type;
	};

	template< typename T >
	struct remove_cvr
	{
		typedef typename remove_cv< typename remove_reference<T>::type >::type
			type;
	};

	template< typename T >
	struct remove_pointer
	{
		typedef T type;
	};

	template< typename T > struct remove_pointer < T* >                { typedef T type; };
	template< typename T > struct remove_pointer < T* const >          { typedef T type; };
	template< typename T > struct remove_pointer < T* volatile >       { typedef T type; };
	template< typename T > struct remove_pointer < T* const volatile > { typedef T type; };

	template< typename T >
	struct remove_extent
	{	
		typedef T type;
	};

	template< typename T, unsigned int N >
	struct remove_extent < T[N] >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_extent < T[] >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_all_extents
	{
		typedef T type;
	};

	template< typename T, unsigned int N >
	struct remove_all_extents < T[N] >
	{
		typedef typename remove_all_extents<T>::type type;
	};

	template< typename T >
	struct remove_all_extents < T[] >
	{
		typedef typename remove_all_extents<T>::type type;
	};

	template< typename T >
	struct add_pointer
	{
		typedef typename remove_reference<T>::type *type;
	};

	namespace detail
	{
		template< typename T, bool CONST, bool VOLATILE>
		struct cv_selector;

		template< typename T >
		struct cv_selector < T, false, false > { typedef T type; };

		template< typename T >
		struct cv_selector < T, true, false > { typedef const T type; };

		template< typename T >
		struct cv_selector < T, false, true > { typedef volatile T type; };

		template< typename T >
		struct cv_selector < T, true, true > { typedef const volatile T type; };

		template< typename SOURCE, typename TARGET,
			bool CONST    = is_const<SOURCE>::value,
			bool VOLATILE = is_volatile<SOURCE>::value >
		struct match_cv
		{
			typedef typename cv_selector< TARGET, CONST, VOLATILE >::type type;
		};

		template < typename T > 
		struct is_void_impl : false_type{};

		template <>
		struct is_void_impl< void > : true_type {};

		template< typename T > 
		struct is_integral_impl : false_type {};

		template<> struct is_integral_impl< bool > : true_type {};
		template<> struct is_integral_impl< char > : true_type {};
		template<> struct is_integral_impl< signed char > : true_type {};
		template<> struct is_integral_impl< unsigned char > : true_type {};
		template<> struct is_integral_impl< wchar_t > : true_type {};
		template<> struct is_integral_impl< signed short > : true_type {};
		template<> struct is_integral_impl< unsigned short > : true_type {};
		template<> struct is_integral_impl< signed int > : true_type {};
		template<> struct is_integral_impl< unsigned int > : true_type {};
		template<> struct is_integral_impl< signed long > : true_type {};
		template<> struct is_integral_impl< unsigned long > : true_type {};
		template<> struct is_integral_impl< signed long long > : true_type{};
		template<> struct is_integral_impl< unsigned long long > : true_type{};

		template< typename T > struct is_floating_point_impl : false_type {};

		template<> struct is_floating_point_impl< float > : true_type{};
		template<> struct is_floating_point_impl< double > : true_type{};
		template<> struct is_floating_point_impl< long double > : true_type{};

		template < typename T > 
		struct signed_type
		{
			typedef T type;
		};

		template<> struct signed_type < char > { typedef signed char type; };
		template<> struct signed_type < unsigned char > { typedef signed char type; };
		template<> struct signed_type < unsigned short > { typedef signed short type; };
		template<> struct signed_type < unsigned int > { typedef signed int type; };
		template<> struct signed_type < unsigned long > { typedef signed long type; };
		template<> struct signed_type < unsigned long long > { typedef signed long long type; };

		template < typename T >
		struct unsigned_type
		{
			typedef T type;
		};

		template<> struct unsigned_type < char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed short > { typedef unsigned short type; };
		template<> struct unsigned_type < signed int > { typedef unsigned int type; };
		template<> struct unsigned_type < signed long > { typedef unsigned long type; };
		template<> struct unsigned_type < signed long long > { typedef unsigned long long type; };

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_signed_impl;

		template < typename T >
		struct make_signed_impl < T, false >
		{
		private:
			typedef signed_type<typename remove_cv<T>::type> signed_type_result;
		public:
			typedef match_cv< T, typename signed_type_result::type > type;
		};

		template < typename T >
		struct make_signed_impl < T, true >
		{
		private:
			static const bool size1test = sizeof( T ) <= sizeof( signed char );
			static const bool size2test = sizeof( T ) <= sizeof( signed short );
			static const bool size4test = sizeof( T ) <= sizeof( signed int );
			typedef typename conditional<size4test, signed int, signed long>::type test4type;
			typedef typename conditional<size2test, signed short, test4type>::type test2type;
		public:
			typedef typename conditional<size1test, signed char, test2type>::type type;
		};

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_unsigned_impl;

		template < typename T >
		struct make_unsigned_impl < T, false >
		{
		private:
			typedef unsigned_type<typename remove_cv<T>::type> unsigned_type_result;
		public:
			typedef match_cv< T, typename unsigned_type_result::type > type;
		};

		template < typename T >
		struct make_unsigned_impl < T, true >
		{
		private:
			static const bool size1test = sizeof( T ) <= sizeof( unsigned char );
			static const bool size2test = sizeof( T ) <= sizeof( unsigned short );
			static const bool size4test = sizeof( T ) <= sizeof( unsigned int );
			typedef typename conditional<size4test, unsigned int, unsigned long>::type test4type;
			typedef typename conditional<size2test, unsigned short, test4type>::type test2type;
		public:
			typedef typename conditional<size1test, unsigned char, test2type>::type type;
		};
	}

	template <typename T> struct is_signed : false_type {};

	template <> struct is_signed<signed char> : true_type {};
	template <> struct is_signed<const signed char> : true_type {};
	template <> struct is_signed<signed short> : true_type {};
	template <> struct is_signed<const signed short> : true_type {};
	template <> struct is_signed<signed int> : true_type {};
	template <> struct is_signed<const signed int> : true_type {};
	template <> struct is_signed<signed long> : true_type {};
	template <> struct is_signed<const signed long> : true_type {};
	template <> struct is_signed<signed long long> : true_type{};
	template <> struct is_signed<const signed long long> : true_type{};

	template <typename T> struct is_unsigned : false_type {};

	template <> struct is_unsigned<unsigned char> : true_type{};
	template <> struct is_unsigned<const unsigned char> : true_type{};
	template <> struct is_unsigned<unsigned short> : true_type{};
	template <> struct is_unsigned<const unsigned short> : true_type{};
	template <> struct is_unsigned<unsigned int> : true_type{};
	template <> struct is_unsigned<const unsigned int> : true_type{};
	template <> struct is_unsigned<unsigned long> : true_type{};
	template <> struct is_unsigned<const unsigned long> : true_type{};
	template <> struct is_unsigned<unsigned long long> : true_type{};
	template <> struct is_unsigned<const unsigned long long> : true_type{};

	template <> struct is_signed<char> : integral_constant<bool, ( char( 0 ) > char( -1 ) ) >{};
	template <> struct is_unsigned<char> : integral_constant<bool, ( char( 0 ) < char( -1 ) ) > {};


	template < typename T >
	struct make_signed
	{
		typedef typename detail::make_signed_impl<T>::type type;
	};

	template <> struct make_signed < bool > ;

	template < typename T >
	struct make_unsigned
	{ 
		typedef typename detail::make_unsigned_impl<T>::type type;
	};
	
	template <> struct make_unsigned < bool > ;

	template< typename T >
	struct is_void : detail::is_void_impl < typename remove_cv<T>::type >
	{
	};

	template< typename T > 
	struct is_integral : detail::is_integral_impl< typename remove_cv<T>::type >
	{
	};

	template< typename T > 
	struct is_floating_point : detail::is_floating_point_impl< typename remove_cv<T>::type >
	{
	};

	template < typename T >
	struct is_arithmetic : integral_constant < bool,
		is_integral<T>::value || is_floating_point<T>::value >
	{
	};

	template <typename T>
	struct is_fundamental : integral_constant < bool,
		is_void<T>::value || is_integral<T>::value || is_floating_point<T>::value >
	{
	};

#if NV_COMPILER == NV_MSVC || NV_COMPILER == NV_CLANG
	template < typename T >
	struct underlying_type
	{
		typedef __underlying_type( T ) type;
	};
#else
	template< typename T >
	struct underlying_type
	{
		typedef typename conditional <
			T( -1 ) < T( 0 ),
			typename make_signed< T >::type,
			typename make_unsigned< T >::type
			> ::type type;
	};
#endif

	template <typename T>
	struct is_empty : integral_constant < bool, __is_empty( T ) > {};

	template < typename T >
	struct is_pod : integral_constant < bool, __is_pod( T ) > {};

	template < typename T >
	struct has_trivial_constructor : integral_constant < bool, __has_trivial_constructor( T ) || __is_pod( T ) > {};

#if NV_COMPILER == NV_MSVC
	template < typename T >
	struct has_trivial_copy : integral_constant < bool, ( __has_trivial_copy( T ) || __is_pod( T ) ) && !is_volatile<T>::value > {};
#else
	template < typename T >
	struct has_trivial_copy : integral_constant < bool, ( __has_trivial_copy( T ) || __is_pod( T ) ) && ( !is_volatile<T>::value && !is_reference<T>::value ) > {};
#endif

	template < typename T >
	struct has_trivial_assign : integral_constant < bool, ( __has_trivial_assign( T ) || __is_pod( T ) ) && ( !is_volatile<T>::value && !is_const<T>::value ) > {};

	template < typename T >
	struct has_trivial_destructor : integral_constant < bool, __has_trivial_destructor( T ) || __is_pod( T ) > {};

	template < typename T >
	struct has_virtual_destructor : integral_constant < bool, __has_virtual_destructor( T ) > {};


	namespace detail
	{

		template < typename F >
		struct function_traits_impl
		{

		};

		template < typename R >
		struct function_traits_impl< R (*)(void) >
		{
			typedef R (*type)();
			typedef R        return_type;
			typedef void     class_type;
			static const int arg_count = 0;
		};

		template < typename R, typename... Args >
		struct function_traits_impl < R(*)(Args...) >
		{
			typedef R(*type)( Args... );
			typedef R        return_type;
			typedef void     class_type;
			static const int arg_count = sizeof...(Args);
		};

		template < class C, typename R >
		struct function_traits_impl< R (C::*)() >
		{
			typedef R (*type)();
			typedef R       return_type;
			typedef C       class_type;
			static const int arg_count = 0;
		};

		template < class C, typename R, typename... Args >
		struct function_traits_impl < R(C::*)(Args...) >
		{
			typedef R(C::*type)(Args...);
			typedef R        return_type;
			typedef C        class_type;
			static const int arg_count = sizeof...(Args);
		};

		template< typename T, typename IS_ENUM >
		struct base_underlying_type_impl
		{
			typedef T type;
		};

		template< typename T >
		struct base_underlying_type_impl < T, true_type >
		{
			typedef typename underlying_type<T>::type type;
		};

		template < typename T > struct is_pointer_impl : false_type {};

		template < typename T > struct is_pointer_impl< T* > : true_type{};
		template < typename T > struct is_pointer_impl< T* const > : true_type{};
		template < typename T > struct is_pointer_impl< T* volatile > : true_type{};
		template < typename T > struct is_pointer_impl< T* const volatile > : true_type{};

	}

	template < typename F >
	struct function_traits : detail::function_traits_impl< F >
	{

	};

	template < typename T >
	struct return_type
	{
		typedef typename function_traits< T >::return_type type;
	};

	template < typename T >
	struct memfn_class_type
	{
		typedef typename function_traits< T >::class_type type;
	};

	template < typename T >
	struct base_underlying_type
	{
		typedef typename detail::base_underlying_type_impl< T, typename is_enum<T>::type >::type type;
	};

	template < typename F > struct is_function_pointer : false_type {};
#if NV_COMPILER == NV_MSVC
	template < typename R, typename... Args > struct is_function_pointer< R( __cdecl * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __stdcall * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __fastcall * )( Args... ) > : true_type{};
	template < typename R, typename... Args > struct is_function_pointer< R( __vectorcall * )( Args... ) > : true_type{};
#else
	template < typename R, typename... Args > struct is_function_pointer< R( *)( Args... ) > : true_type{};
#endif

	template < typename C > struct is_member_function_pointer : false_type {};
#if NV_COMPILER == NV_MSVC
#define NV_IS_MEMFNPTR( call_conv ) \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) const > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) volatile > : true_type{}; \
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( call_conv C::* )( Args... ) const volatile > : true_type{};

NV_IS_MEMFNPTR( __thiscall )
NV_IS_MEMFNPTR( __cdecl )
NV_IS_MEMFNPTR( __stdcall )
NV_IS_MEMFNPTR( __fastcall )
NV_IS_MEMFNPTR( __vectorcall )

#undef NV_IS_MEMFNPTR
#else
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) const > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) volatile > : true_type{};
	template < typename R, typename C, typename... Args > struct is_member_function_pointer< R( C::* )( Args... ) const volatile > : true_type{};
#endif

	template < typename F >	struct is_member_pointer : integral_constant < bool, is_member_function_pointer<F>::value > {};
	template < typename C, typename R > struct is_member_pointer<R C::*> : true_type {};

	template < typename T >
	struct is_pointer : integral_constant< bool, (detail::is_pointer_impl<T>::value) && !(is_member_pointer<T>::value) > {};

	template< typename T >
	struct is_function : integral_constant< bool, is_function_pointer< typename remove_cv<T>::type *>::value > {};

	template< typename T >
	struct is_function < T& > : false_type {};

	template< typename T >
	struct is_function < T&& > : false_type {};

	template< typename T >
	struct decay
	{
		typedef typename remove_reference<T>::type U;
		typedef typename conditional <
			is_array<U>::value,
			typename remove_extent<U>::type*,
			typename conditional <
				is_function<U>::value,
				typename add_pointer<U>::type,
				typename remove_cv<U>::type
			> ::type
		> ::type type;
	};

#if NV_COMPILER == NV_MSVC
	typedef double max_align_t;
#elif NV_PLATFORM == NV_APPLE
	typedef long double max_align_t;
#else
	namespace detail
	{
		class aligned_dummy;
		typedef void( *aligned_fptr )( );
		typedef int( aligned_dummy::*aligned_memptr );
		typedef int ( aligned_dummy::*aligned_memfptr )( );
	}

	union max_align_t
	{
		double                  dummy0;
		long double             dummy1;
		void*                   dummy2;
		long long               dummy3;
		detail::aligned_fptr    dummy4;
		detail::aligned_memptr  dummy5;
		detail::aligned_memfptr dummy6;
	};
#endif

#if NV_COMPILER == NV_CLANG
	template< typename T, size_t Size, size_t Align >
	struct aligned_array
	{
		typedef T alignas( Align ) type[Size];
	};
#elif NV_COMPILER == NV_GNUC
 	template< typename T, size_t Size, size_t Align >
 	struct aligned_array
 	{
		typedef T __attribute__((aligned( Align ))) type[Size];
	};
#else
	// TODO: remove this shit after moving to MSVC 2015
	template< typename T, size_t Size, size_t Align >
	struct aligned_array;

	// According to LLVM ( https://llvm.org/svn/llvm-project/llvm/trunk/include/llvm/Support/AlignOf.h )
	// MSVC has problems with align below 16...
#define NV_ALIGNED_ARRAY(N) \
	template< typename T, size_t Size > \
	struct aligned_array< T, Size, N >  \
	{ \
		typedef __declspec( align(N) ) T type[Size]; \
	}; 

NV_ALIGNED_ARRAY( 1 )
NV_ALIGNED_ARRAY( 2 )
NV_ALIGNED_ARRAY( 4 )
NV_ALIGNED_ARRAY( 8 )
NV_ALIGNED_ARRAY( 16 )
NV_ALIGNED_ARRAY( 32 )
NV_ALIGNED_ARRAY( 64 )
NV_ALIGNED_ARRAY( 128 )

#undef NV_ALIGNED_ARRAY
#endif

	template< size_t Size, size_t Align = NV_ALIGN_OF( max_align_t ) >
	struct aligned_storage
	{
		struct type
		{
			typename aligned_array< unsigned char, Size, Align >::type data;
		};
	};

}

#endif // NV_STL_TYPE_TRAITS_HH
