// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file algorithm.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief STL algorithm library
*/
// TODO: implement fill_default!

#ifndef NV_STL_ALGORITHM_HH
#define NV_STL_ALGORITHM_HH

#include <nv/core/common.hh>
#include <nv/stl/utility.hh>
#include <nv/stl/iterator.hh>

namespace nv
{

	template < typename InputIterator, typename EqualityComparable >
	inline typename iterator_traits<InputIterator>::difference_type 
	count( InputIterator first, InputIterator last, const EqualityComparable& value )
	{
		typename iterator_traits<InputIterator>::difference_type result = 0;
		while ( first != last )
		{
			if ( *first == value ) ++result;
			++first;
		}
		return result;
	}

	template < typename InputIterator, typename EqualityComparable >
	inline InputIterator find( InputIterator first, InputIterator last, const EqualityComparable& val )
	{
		while ( first != last )
		{
			if ( *first == val ) return first;
			++first;
		}
		return last;
	}

	template < typename InputIterator, typename UnaryPredicate >
	inline InputIterator find_if( InputIterator first, InputIterator last, UnaryPredicate find_op )
	{
		for ( ; first != last; ++first )
			if ( find_op( *first ) )
				break;
		return first;
	}

	template < typename InputIterator, typename ForwardIterator >
	inline InputIterator find_first_of( InputIterator first1, InputIterator last1, ForwardIterator first2, ForwardIterator last2 )
	{
		for ( ; first1 != last1; ++first1 )
			for ( ForwardIterator it = first2; it != last2; ++it )
				if ( *first1 == *it )
					break;
		return ( first1 );
	}

	template < typename InputIterator, typename ForwardIterator, typename BinaryPredicate >
	inline InputIterator find_first_of( InputIterator first1, InputIterator last1, ForwardIterator first2, ForwardIterator last2, BinaryPredicate comp_op )
	{
		for ( ; first1 != last1; ++first1 )
			for ( ForwardIterator it = first2; it != last2; ++it )
				if ( comp_op( *first1, *it ) )
					break;
		return ( first1 );
	}

	template < typename ForwardIterator1, typename ForwardIterator2, typename BinaryPredicate >
	inline ForwardIterator1 search( ForwardIterator1 first1, ForwardIterator1 last1, ForwardIterator2 first2, ForwardIterator2 last2, BinaryPredicate comp_op )
	{
		if ( first2 == last2 ) return first1;
		while ( first1 != last1 )
		{
			ForwardIterator1 it1 = first1;
			ForwardIterator2 it2 = first2;
			while ( comp_op( *it1, *it2 ) )
			{
				++it1; ++it2;
				if ( it2 == last2 ) return first1;
				if ( it1 == last1 ) return last1;
			}
			++first1;
		}
		return last1;
	}

	template < typename ForwardIterator1, typename ForwardIterator2 >
	inline ForwardIterator1 search( ForwardIterator1 first1, ForwardIterator1 last1, ForwardIterator2 first2, ForwardIterator2 last2 )
	{
		if ( first2 == last2 ) return first1; 
		while ( first1 != last1 )
		{
			ForwardIterator1 it1 = first1;
			ForwardIterator2 it2 = first2;
			while ( *it1 == *it2 )
			{
				++it1; ++it2;
				if ( it2 == last2 ) return first1;
				if ( it1 == last1 ) return last1;
			}
			++first1;
		}
		return last1;
	}

	template < typename InputIterator, typename OutputIterator, typename UnaryOperator >
	inline OutputIterator transform_seq( InputIterator first1, InputIterator last1, OutputIterator result, UnaryOperator op )
	{
		for ( ; first1 != last1; ++result, ++first1 )
			*result = op( *first1 );
		return result;
	}

	template < typename InputIterator1, typename InputIterator2, typename OutputIterator, typename BinaryOperator >
	inline OutputIterator transform_seq( InputIterator1 first1, InputIterator1 last1, InputIterator2 first2, OutputIterator result, BinaryOperator binary_op )
	{
		for ( ; first1 != last1; ++result, ++first1, ++first2 )
			*result = binary_op( *first1, *first2 );
		return result;
	}

	template < typename ForwardIterator, typename UnaryPredicate >
	inline ForwardIterator remove_if( ForwardIterator first, ForwardIterator last, UnaryPredicate pred )
	{
		ForwardIterator result = first;
		while ( first != last )
		{
			if ( !pred( *first ) )
			{
				*result = move( *first );
				++result;
			}
			++first;
		}
		return result;
	}

	template < typename Iterator >
	inline void iter_swap( Iterator a, Iterator b )
	{
		swap( *a, *b );
	}

	template <typename BidirectionalIterator>
	inline void reverse( BidirectionalIterator first, BidirectionalIterator last )
	{
		for ( ; distance( first, --last ) > 0; ++first )
			iter_swap( first, last );
	}

	template < typename ForwardIterator >
	ForwardIterator rotate( ForwardIterator first, ForwardIterator middle, ForwardIterator last )
	{
		if ( first == middle || middle == last )
			return first;
		reverse( first, middle );
		reverse( middle, last );
		for ( ; first != middle && middle != last; ++first )
			iter_swap( first, --last );
		reverse( first, ( first == middle ? last : middle ) );
		return first;
	}

	// TODO: specialize for rotate on pointers!
	//template <typename T>
	//inline T* rotate( T* first, T* middle, T* last );

	template < typename RandomAccessIterator, typename Compare >
	void stable_sort( RandomAccessIterator first, RandomAccessIterator last, Compare comp )
	{
		for ( RandomAccessIterator j, i = first; ++i < last; )
		{
			for ( j = i; j-- > first && comp( *i, *j ); );
			if ( ++j != i ) rotate( j, i, i + 1 );
		}
	}

	namespace detail
	{
		template < typename ForwardIterator, typename T >
		void fill_impl( ForwardIterator first, ForwardIterator last, const T& value, false_type )
		{
			for (; first != last; ++first )
				*first = value;
		}

		template < typename ForwardIterator, typename T >
		void fill_impl( ForwardIterator first, ForwardIterator last, const T& value, true_type )
		{
			memset( first, value, last - first );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator fill_n_impl( ForwardIterator first, size_t count, const T& value, false_type )
		{
			for ( ; count-- > 0; ++first ) *first = value;
			return first;
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator fill_n_impl( ForwardIterator first, size_t count, const T& value, true_type )
		{
			return ForwardIterator( memset( first, value, count ) + count );
		}
	}

	// TODO: fill_default special case!
	template < typename ForwardIterator, typename T > 
	inline void fill( ForwardIterator first, ForwardIterator last, const T& value )
	{
		detail::fill_impl( first, last, value,
			bool_constant< sizeof( T ) == 1 && has_trivial_assign<T>::value >() );
	}

	template < typename ForwardIterator, typename T >
	inline ForwardIterator fill_n( ForwardIterator first, size_t count, const T& value )
	{
		return detail::fill_n_impl( first, count, value,
			bool_constant< sizeof( T ) == 1 && has_trivial_assign<T>::value >() );
	}

	template < typename ForwardIterator >
	inline void fill_default( ForwardIterator first, ForwardIterator last )
	{
		// TODO: implement
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::fill_impl( first, last, value_type(),
			bool_constant< sizeof( value_type ) == 1 && has_trivial_assign<value_type>::value >() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator fill_default_n( ForwardIterator first, size_t count )
	{
		// TODO: implement
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::fill_n_impl( first, count, value_type(),
			bool_constant< sizeof( value_type ) == 1 && has_trivial_assign<value_type>::value >() );
	}

}

#endif // NV_STL_ALGORITHM_HH