// Copyright (C) 2012-2014 ChaosForge Ltd
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include <cstdio>
#include <sys/stat.h> 
#include "nv/io/c_stream.hh"

using namespace nv;

c_stream::c_stream()
	: m_file( nullptr )
	, m_file_name( "" )
	, m_file_size( size_t(-1) )
{

}

c_stream::c_stream( void* pfile, const char* filename )
	: m_file( pfile )
	, m_file_name( filename )
	, m_file_size( size_t(-1) )
{

}

c_stream::~c_stream()
{
	if ( m_file )
	{
		::fclose( (FILE*)m_file );
	}
}

size_t c_stream::read( void* buffer, size_t size, size_t count )
{
	NV_ASSERT( buffer != nullptr && size != 0 && count != 0, "Bad parameter passed to read!" );
	return m_file ? ::fread( buffer, size, count, (FILE*)m_file ) : 0;
}

size_t c_stream::write( const void* buffer, size_t size, size_t count )
{
	NV_ASSERT( buffer != nullptr && size != 0 && count != 0, "Bad parameter passed to write!" );
	return m_file ? ::fwrite( buffer, size, count, (FILE*)m_file ) : 0;
}

bool c_stream::seek( long offset, origin orig )
{
	return m_file != nullptr && ( ::fseek( (FILE*)m_file, (long)offset, static_cast<int>(orig) ) == 0 );
}

size_t c_stream::tell()
{
	return m_file != nullptr ? static_cast< size_t >( ::ftell( (FILE*)m_file ) ) : 0;
}

size_t c_stream::size()
{
	if ( m_file == nullptr || m_file_name == nullptr )
	{
		return 0;
	}

	if ( m_file_size == size_t(-1) )
	{
		struct stat fstat; 
		int result = stat(m_file_name, &fstat ); 
		if ( result != 0 ) 
		{
			return 0; 
		}
		m_file_size = (size_t) (fstat.st_size); 		
	}

	return m_file_size;
}

void c_stream::flush()
{
	if ( m_file != nullptr )
	{
		::fflush( (FILE*)m_file );
	}
}
