// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file type_traits.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type traits
 */
// TODO: function_traits support only up to 4 parameters because:
//    -- if you have more than 4 params, you're doing something wrong
//    -- once the Variadic Templates cometh, for great justice we will win!

#ifndef NV_CORE_TYPE_TRAITS_HH
#define NV_CORE_TYPE_TRAITS_HH

#include <nv/core/common.hh>
#include <type_traits>
#include <typeinfo>

namespace nv
{

	template< typename T, T VALUE>
	struct integral_constant
	{
		static const T value = VALUE;
		typedef T value_type;
		typedef integral_constant<T, VALUE> type;
		operator value_type() const { return ( value ); }
	};

	typedef integral_constant<bool, true> true_type;
	typedef integral_constant<bool, false> false_type;

	template< bool TEST, typename T = void>
	struct enable_if {};

	template< typename T > 
	struct enable_if< true, T >
	{
		typedef T type;
	};

	template< bool TEST, typename T1, typename T2 >
	struct conditional
	{
		typedef T2 type;
	};

	template< typename T1, typename T2>
	struct conditional < true, T1, T2 >
	{
		typedef T1 type;
	};

	template< typename T1, typename T2 >
	struct is_same : false_type {};

	template< typename T >
	struct is_same < T, T > : true_type{};

	template< typename T >
	struct is_lvalue_reference : false_type {};

	template< typename T >
	struct is_lvalue_reference < T& > : true_type{};

	template< typename T >
	struct is_rvalue_reference : false_type {};

	template< typename T >
	struct is_rvalue_reference < T&& > : true_type{};

	template < typename T >
	struct is_enum : integral_constant < bool, __is_enum( T ) > {};

	template< typename T >
	struct remove_reference
	{
		typedef T type;
	};

	template< typename T >
	struct remove_reference < T& >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_reference < T&& >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const < const T >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_const < const T[] >
	{
		typedef T type[];
	};

	template< typename T, unsigned int N >
	struct remove_const < const T[N] >
	{
		typedef T type[N];
	};

	template< typename T >
	struct remove_volatile
	{
		typedef T type;
	};

	template< typename T >
	struct remove_volatile < volatile T >
	{
		typedef T type;
	};

	template< typename T >
	struct remove_volatile < volatile T[] >
	{
		typedef T type[];
	};

	template< typename T, unsigned int N >
	struct remove_volatile < volatile T[N] >
	{
		typedef T type[N];
	};

	template< typename T >
	struct remove_cv
	{
		typedef typename remove_const< typename remove_volatile<T>::type >::type
			type;
	};

	namespace detail
	{
		// TODO: these seem to simple compared to MSVC/GCC - so we'll leave them in
		// detail - research why it is so.
		template < typename T >
		struct is_const : public false_type {};
		template < typename T >
		struct is_const < T const > : public true_type {};
		template < typename T > 
		struct is_volatile : public false_type {};
		template < typename T >
		struct is_volatile < T volatile > : public true_type {};
		// TODO END

		template< typename T, bool CONST, bool VOLATILE>
		struct cv_selector;

		template< typename T >
		struct cv_selector < T, false, false > { typedef T type; };

		template< typename T >
		struct cv_selector < T, true, false > { typedef const T type; };

		template< typename T >
		struct cv_selector < T, false, true > { typedef volatile T type; };

		template< typename T >
		struct cv_selector < T, true, true > { typedef const volatile T type; };

		template< typename SOURCE, typename TARGET,
			bool CONST    = is_const<SOURCE>::value,
			bool VOLATILE = is_volatile<SOURCE>::value >
		struct match_cv
		{
			typedef typename cv_selector< TARGET, CONST, VOLATILE >::type type;
		};

		template< typename T > 
		struct is_integral_impl : false_type {};

		template<> struct is_integral_impl< bool > : true_type {};
		template<> struct is_integral_impl< char > : true_type {};
		template<> struct is_integral_impl< signed char > : true_type {};
		template<> struct is_integral_impl< unsigned char > : true_type {};
		template<> struct is_integral_impl< wchar_t > : true_type {};
		template<> struct is_integral_impl< signed short > : true_type {};
		template<> struct is_integral_impl< unsigned short > : true_type {};
		template<> struct is_integral_impl< signed int > : true_type {};
		template<> struct is_integral_impl< unsigned int > : true_type {};
		template<> struct is_integral_impl< signed long > : true_type {};
		template<> struct is_integral_impl< unsigned long > : true_type {};
		template<> struct is_integral_impl< sint64 > : true_type {};
		template<> struct is_integral_impl< uint64 > : true_type {};

		template< typename T > struct is_floating_point_impl : false_type {};

		template<> struct is_floating_point_impl< float > : true_type{};
		template<> struct is_floating_point_impl< double > : true_type{};
		template<> struct is_floating_point_impl< long double > : true_type{};

		template < typename T > 
		struct signed_type
		{
			typedef T type;
		};

		template<> struct signed_type < char > { typedef signed char type; };
		template<> struct signed_type < unsigned char > { typedef signed char type; };
		template<> struct signed_type < unsigned short > { typedef signed short type; };
		template<> struct signed_type < unsigned int > { typedef signed int type; };
		template<> struct signed_type < unsigned long > { typedef signed long type; };
		template<> struct signed_type < uint64 > { typedef sint64 type; };

		template < typename T >
		struct unsigned_type
		{
			typedef T type;
		};

		template<> struct unsigned_type < char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed char > { typedef unsigned char type; };
		template<> struct unsigned_type < signed short > { typedef unsigned short type; };
		template<> struct unsigned_type < signed int > { typedef unsigned int type; };
		template<> struct unsigned_type < signed long > { typedef unsigned long type; };
		template<> struct unsigned_type < sint64 > { typedef uint64 type; };

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_signed_impl;

		template < typename T >
		struct make_signed_impl < T, false >
		{
		private:
			typedef signed_type<typename remove_cv<T>::type> signed_type_result;
		public:
			typedef match_cv< T, typename signed_type_result::type > type;
		};

		template < typename T >
		struct make_signed_impl < T, true >
		{
		private:
			static const bool size1test = sizeof( T ) <= sizeof( signed char );
			static const bool size2test = sizeof( T ) <= sizeof( signed short );
			static const bool size4test = sizeof( T ) <= sizeof( signed int );
			typedef typename conditional<size4test, signed int, signed long>::type test4type;
			typedef typename conditional<size2test, signed short, test4type>::type test2type;
		public:
			typedef typename conditional<size1test, signed char, test2type>::type type;
		};

		template < typename T, bool IS_ENUM = is_enum< T >::value >
		struct make_unsigned_impl;

		template < typename T >
		struct make_unsigned_impl < T, false >
		{
		private:
			typedef unsigned_type<typename remove_cv<T>::type> unsigned_type_result;
		public:
			typedef match_cv< T, typename unsigned_type_result::type > type;
		};

		template < typename T >
		struct make_unsigned_impl < T, true >
		{
		private:
			static const bool size1test = sizeof( T ) <= sizeof( unsigned char );
			static const bool size2test = sizeof( T ) <= sizeof( unsigned short );
			static const bool size4test = sizeof( T ) <= sizeof( unsigned int );
			typedef typename conditional<size4test, unsigned int, unsigned long>::type test4type;
			typedef typename conditional<size2test, unsigned short, test4type>::type test2type;
		public:
			typedef typename conditional<size1test, unsigned char, test2type>::type type;
		};
	}

	template < typename T >
	struct make_signed
	{
		typedef typename detail::make_signed_impl<T>::type type;
	};

	template <> struct make_signed < bool > ;

	template < typename T >
	struct make_unsigned
	{ 
		typedef typename detail::make_unsigned_impl<T>::type type;
	};
	
	template <> struct make_unsigned < bool > ;


	template< typename T > 
	struct is_integral : detail::is_integral_impl< typename remove_cv<T>::type >
	{
	};

	template< typename T > 
	struct is_floating_point : detail::is_floating_point_impl< typename remove_cv<T>::type >
	{
	};

#if NV_COMPILER == NV_MSVC || NV_COMPILER == NV_CLANG
	template < typename T >
	struct underlying_type
	{
		typedef __underlying_type( T ) type;
	};
#else
	template< typename T >
	struct underlying_type
	{
		typedef typename conditional <
			T( -1 ) < T( 0 ),
			typename make_signed< T >::type,
			typename make_unsigned< T >::type
			> ::type type;
	};
#endif

	namespace detail
	{

		template < typename F >
		struct function_traits_impl
		{

		};

		template < typename R >
		struct function_traits_impl< R (*)(void) >
		{
			typedef R (*type)();
			typedef R        return_type;
			typedef void     class_type;
			static const int arg_count = 0;
		};

		template < typename R, typename... Args >
		struct function_traits_impl < R(*)(Args...) >
		{
			typedef R(*type)( Args... );
			typedef R        return_type;
			typedef void     class_type;
			static const int arg_count = sizeof...(Args);
		};

		template < class C, typename R >
		struct function_traits_impl< R (C::*)() >
		{
			typedef R (*type)();
			typedef R       return_type;
			typedef C       class_type;
			static const int arg_count = 0;
		};

		template < class C, typename R, typename... Args >
		struct function_traits_impl < R(C::*)(Args...) >
		{
			typedef R(C::*type)(Args...);
			typedef R        return_type;
			typedef C        class_type;
			static const int arg_count = sizeof...(Args);
		};

	}

	template < typename F >
	struct function_traits : public detail::function_traits_impl< F >
	{

	};

	template < typename T >
	struct return_type
	{
		typedef typename function_traits< T >::return_type type;
	};

	template < typename T >
	struct memfn_class_type
	{
		typedef typename function_traits< T >::class_type type;
	};

	template <typename TYPE> 
	void construct_object(void* object)
	{
		new (object) TYPE;
	}
	template <typename TYPE> 
	void destroy_object(void* object)
	{
		((TYPE*)object)->TYPE::~TYPE();
	}

	template< typename T, typename IS_ENUM >
	struct base_underlying_type_helper
	{
		typedef T type;
	};

	template< typename T >
	struct base_underlying_type_helper< T, true_type >
	{
		typedef typename underlying_type<T>::type type;
	};


	template< typename T >
	struct base_underlying_type
	{
		typedef typename base_underlying_type_helper< T, typename is_enum<T>::type >::type type;
	};

	class type_index
	{
	public:
		inline type_index( const std::type_info& info ) : m_type_info( &info ) {}
		inline size_t hash_code() const { return m_type_info->hash_code(); }
		inline const char *name() const { return m_type_info->name(); }

		inline bool operator==( const type_index& rhs ) const
		{
			return ( *m_type_info == *rhs.m_type_info );
		}

		inline bool operator!=( const type_index& rhs ) const
		{
			return ( !( *this == rhs ) );
		}

		inline bool operator<( const type_index& rhs ) const
		{
			return ( m_type_info->before( *rhs.m_type_info ) );
		}

		inline bool operator>=( const type_index& rhs ) const
		{
			return ( !( *this < rhs ) );
		}

		inline bool operator>( const type_index& rhs ) const
		{
			return ( rhs < *this );
		}

		inline bool operator<=( const type_index& rhs ) const
		{
			return ( !( rhs < *this ) );
		}

	private:
		const std::type_info* m_type_info;
	};

}

#endif // NV_CORE_TYPE_TRAITS_HH
