// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file cstring_store.hh
* @author Kornel Kisielewicz
* @brief cstring_store
*/

#ifndef NV_CORE_CSTRING_STORE_HH
#define NV_CORE_CSTRING_STORE_HH

#include <nv/core/common.hh>
#include <nv/core/array.hh>
#include <unordered_map>
#include <cstring>
#include <cstdlib>

namespace nv
{

	namespace detail
	{

		inline char *cstring_duplicate( const char *s )
		{
			size_t length = std::strlen( s ) + 1;
			void *result = std::malloc( length );
			if ( result == nullptr ) return nullptr;
			return (char *)std::memcpy( result, s, length );
		}

		struct cstring_compare
		{
			bool operator()( const char* lhs, const char* rhs ) const
			{
				return strcmp( lhs, rhs ) == 0;
			}
		};


		struct cstring_hash
		{
			int operator()( const char* str ) const
			{
				int seed = 131;
				int hash = 0;
				while ( *str )
				{
					hash = ( hash * seed ) + ( *str );
					str++;
				}
				return hash & ( 0x7FFFFFFF );
			}
		};
	}

	template < typename T >
	using cstring_unordered_map = std::unordered_map < const char*, T, detail::cstring_hash, detail::cstring_compare > ;

	struct cstring_deleter : public noncopyable
	{
	public:
		cstring_deleter() {}

		char* duplicate( const char* s )
		{
			char* result = detail::cstring_duplicate( s );
			insert( result );
			return result;
		}

		void insert( char * s )
		{
			m_array.push_back( s );
		}

		~cstring_deleter()
		{
			for ( auto s : m_array ) free( s );
		}
	private:
		std::vector< char* > m_array;
	};

	class cstring_store : public noncopyable
	{
	public:
		cstring_store() {}

		bool exists( const char* cstr )
		{
			return m_map.find( cstr ) != std::end( m_map );
		}
		
		const char* get( uint32 index )
		{
			return index < m_array.size() ? m_array[index] : nullptr;
		}

		uint32 resolve( const char* cstr )
		{
			auto it = m_map.find( cstr );
			if ( it != std::end( m_map ) )
			{
				return it->second;
			}
			return uint32(-1);
		}

		uint32 push( const char* cstr )
		{
			char* duplicate = detail::cstring_duplicate( cstr );
			m_array.push_back( duplicate );
			uint32 index = (uint32)m_array.size() - 1;
			m_map[duplicate] = index;
			return index;
		}

		uint32 insert( const char* cstr )
		{
			auto it = m_map.find( cstr );
			if ( it == std::end( m_map ) )
			{
				return push( cstr );
			}
			return it->second;
		}



		~cstring_store()
		{
			for ( auto s : m_array ) free( s );
		}
	private:
		std::vector< char* >            m_array;
		cstring_unordered_map< uint32 > m_map;
	};

}

#endif // NV_CORE_STRING_STORE_HH
