// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/fmod/fmod_audio.hh"

#include "nv/lib/fmod.hh"
#include "nv/core/logging.hh"

using namespace nv;

fmod::audio::audio()
{
	m_system = nullptr;

	nv::load_fmod_library();

	FMOD_RESULT result;
	FMOD_SYSTEM* system;

	result = FMOD_System_Create( &system );
	if ( result != FMOD_OK )
	{
		NV_LOG( LOG_CRITICAL, "Failed to create FMOD System -- " << FMOD_ErrorString( result ) );
		return;
	}
	result = FMOD_System_Init( system, 64, FMOD_INIT_NORMAL, 0 );
	if ( result != FMOD_OK )
	{
		NV_LOG( LOG_ERROR, "Failed to initialize FMOD System -- " << FMOD_ErrorString( result ) );
		return;
	}
	m_system = system;
}


nv::channel fmod::audio::play_sound( nv::sound a_sound )
{
	sound_info* info = m_sounds.get( a_sound );
	if ( info )
	{
		FMOD_SYSTEM* system = (FMOD_SYSTEM*)m_system;
		FMOD_SOUND* sample  = (FMOD_SOUND*)( info->fmod_sound );
		FMOD_RESULT result  = FMOD_System_PlaySound( system, FMOD_CHANNEL_FREE, sample, false, 0 );
		if ( result != FMOD_OK )
		{
			NV_LOG( LOG_WARNING, "FMOD failed to play sound -- " << FMOD_ErrorString( result ) );
		}
	}
	return channel();
}

nv::sound fmod::audio::load_sound( const std::string& a_path )
{
	FMOD_SYSTEM* system = (FMOD_SYSTEM*)m_system;
	FMOD_SOUND* sample;
	FMOD_RESULT fm_result = FMOD_System_CreateSound( system, a_path.c_str(), FMOD_DEFAULT, 0, &sample );
	if ( fm_result != FMOD_OK )
	{
		NV_LOG( LOG_ERROR, "FMOD failed to load sample '" << a_path << "' -- " << FMOD_ErrorString( fm_result ) );
		return sound();
	}
	sound result = m_sounds.create();
	sound_info* info = m_sounds.get( result );
	info->fmod_sound = sample;
	return result;
}

void nv::fmod::audio::release( sound a_sound )
{
	sound_info* info = m_sounds.get( a_sound );
	if ( info )
	{
		FMOD_Sound_Release( (FMOD_SOUND*)info->fmod_sound );
		m_sounds.destroy( a_sound );
	}
}

void fmod::audio::update()
{
	FMOD_System_Update( (FMOD_SYSTEM*)m_system );
	// no-op
}

fmod::audio::~audio()
{
	while ( m_sounds.size() > 0 )
		release( m_sounds.get_handle(0) );
	FMOD_System_Release( (FMOD_SYSTEM*)m_system );
}

