// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gui/gui_environment.hh"

#include "nv/gui/gui_renderer.hh"


	/*

	TODO: parse a lua stylesheet as per Trac wiki

	IDEA: Store everything in std::unordered_maps, with lua_value's?

	A lua_value is a variant stores strings as const char* that deletes them on destructor?
	Question is that there might not be much gained on speed anyway, due to Lua's speed.
	Special function field allows delayed per parse execution?

	*/

#include "nv/gfx/sliced_buffer.hh"
#include "nv/gfx/texture_atlas.hh"

nv::gui::environment::environment( window* w )
	: m_renderer( nullptr ), m_window( w )
{
	m_area.dim( dimension( w->get_width(), w->get_height() ) );
	
	m_screen           = m_elements.create();
	element* screen    = m_elements.get( m_screen );
	screen->m_absolute = m_area;
	screen->m_relative = m_area;

	m_renderer = new renderer( w );
}

void nv::gui::environment::load_style( const std::string& filename )
{
	m_renderer->load_style( filename );
}

nv::gui::handle nv::gui::environment::create_element( const rectangle& r )
{
	return create_element( m_screen, r );
}

nv::gui::handle nv::gui::environment::create_element( handle parent, const rectangle& r )
{
	if ( parent.is_nil() ) parent = m_screen;
	handle result = m_elements.create();
	element* e    = m_elements.get( result );
	e->m_absolute = r;
	e->m_relative = r;
	add_child( parent, result );
	return result;
}

void nv::gui::environment::destroy_element( handle e )
{
	element* dead_element = m_elements.get( e );
	if ( dead_element == nullptr ) return;
	destroy_children( e );
	remove_child( dead_element->m_parent, e );

	delete dead_element->m_render_data; 
	dead_element->m_render_data = nullptr;
	dead_element->m_parent = handle();

	m_elements.destroy( e );
}

void nv::gui::environment::update( handle e, uint32 elapsed )
{
	element* el = m_elements.get( e );
	if ( !el ) return;
	//	el->on_update( elapsed );
	if ( el->m_visible )
	{
		for ( handle i : el->m_children )
		{
			update( i, elapsed );
		}
	}
	if ( el->m_dirty || el->m_render_data == nullptr )
	{
		m_renderer->redraw( el, elapsed );
		el->m_dirty = false;
	}
}

void nv::gui::environment::draw( handle e )
{
	element* el = m_elements.get( e );
	if ( !el ) return;
	if ( el->m_visible )
	{
//		el->on_draw();
		m_renderer->draw( el );
		for ( handle i : el->m_children )
		{
			draw(i);
		}
	}
}

void nv::gui::environment::update()
{
	update( m_screen, 0 );
}

void nv::gui::environment::draw()
{
	draw( m_screen );
	m_renderer->draw();
}

void nv::gui::environment::add_child( handle child )
{
	add_child( m_screen, child );
}

void nv::gui::environment::add_child( handle parent, handle child )
{
	element* e = m_elements.get( child );
	element* p = m_elements.get( parent );
	if ( e && p )
	{
		remove_child( e->m_parent, child );
		e->m_parent = parent;
		p->m_children.push_back( child );
		p->m_child_count++;
	}
}

void nv::gui::environment::destroy_children( handle e )
{
	element* parent = m_elements.get(e);
	if ( parent )
	{
		while ( !parent->m_children.empty() )
		{
			destroy_element( parent->m_children.front() );
		}
	}
}


nv::gui::environment::~environment()
{
	destroy_children( handle() );
	delete m_renderer;
}

bool nv::gui::environment::process_io_event( const io_event& )
{
	return false;
}

bool nv::gui::environment::process_io_event( handle e, const io_event& ev )
{
	element* el = m_elements.get( e );
	return el && el->m_parent.is_valid() ? process_io_event( el->m_parent, ev ) : false;
}

nv::gui::handle nv::gui::environment::get_element( const position& p )
{
	return get_deepest_child( m_screen, p );
}

nv::gui::handle nv::gui::environment::get_deepest_child( handle e, const position& p )
{
	element* el = m_elements.get(e);
	if ( !el && !el->m_visible ) return handle();

	handle result;
	element::list::reverse_iterator it = el->m_children.rbegin();

	while ( it != el->m_children.rend() )
	{
		result = get_deepest_child( *it, p );
		if ( result.is_valid() ) return result;
		++it;
	}

	if ( el->m_absolute.contains(p) ) return e;
	return handle();
}

void nv::gui::environment::move_to_top( handle child )
{
	element* e      = m_elements.get( child );
	element* parent = m_elements.get( e->m_parent );
	if ( e && parent )
	{
		auto it = std::find( parent->m_children.begin(), parent->m_children.end(), child );
		if ( it != parent->m_children.end() )
		{
			parent->m_children.erase( it );
			parent->m_children.push_back( child );
			parent->m_dirty = true;
		}	
	}
}

void nv::gui::environment::move_to_bottom( handle child )
{
	element* e      = m_elements.get( child );
	element* parent = m_elements.get( e->m_parent );
	if ( e && parent )
	{
		auto it = std::find( parent->m_children.begin(), parent->m_children.end(), child );
		if ( it != parent->m_children.end() )
		{
			parent->m_children.erase( it );
			parent->m_children.push_front( child );
			parent->m_dirty = true;
		}
	}
}

void nv::gui::environment::set_relative( handle e, const rectangle& r )
{
	element* el = m_elements.get(e);
	if ( el )
	{
		el->m_dirty    = true;
		el->m_relative = r;
		recalculate_absolute( e );
	}
}

void nv::gui::environment::set_relative( handle e, const position& p )
{
	element* el = m_elements.get(e);
	if ( el )
	{
		set_relative( e, rectangle( p, p + el->m_relative.get_size() ) );
	}
}

void nv::gui::environment::recalculate_absolute( handle e )
{
	element* el = m_elements.get(e);
	rectangle pabsolute = m_elements.get( el->m_parent )->m_absolute;
	el->m_absolute = el->m_relative + pabsolute.ul;

	for ( handle o : el->m_children )
	{
		recalculate_absolute( o );
	}
}

void nv::gui::environment::set_class( handle e, const string& text )
{
	element* ep = m_elements.get(e);
	if ( ep != nullptr )
	{
		ep->m_class = text;
		ep->m_dirty = true;
	}
}

void nv::gui::environment::set_text( handle e, const string& text )
{
	element* ep = m_elements.get(e);
	if ( ep != nullptr )
	{
		ep->m_text = text;
		ep->m_dirty = true;
	}
}

void nv::gui::environment::remove_child( handle parent, handle child )
{
	element* p = m_elements.get( parent );
	if ( p )
	{
		auto it = std::find( p->m_children.begin(), p->m_children.end(), child );
		if ( it != p->m_children.end() )
		{
			element* e = m_elements.get( *it );
			e->m_parent = handle();
			p->m_children.erase(it);
		}	
	}
}

