// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_MATH_HH
#define NV_MATH_HH

#include <nv/common.hh>

#if NV_COMPILER == NV_GNUC
#pragma GCC system_header
#elif NV_COMPILER == NV_CLANG
#pragma clang system_header
#endif

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include <glm/gtc/quaternion.hpp>

namespace nv
{

	typedef glm::detail::tvec2<sint8> i8vec2;
	typedef glm::detail::tvec3<sint8> i8vec3;
	typedef glm::detail::tvec4<sint8> i8vec4;

	typedef glm::detail::tvec2<sint16> i16vec2;
	typedef glm::detail::tvec3<sint16> i16vec3;
	typedef glm::detail::tvec4<sint16> i16vec4;

	typedef glm::vec2 vec2;
	typedef glm::vec3 vec3;
	typedef glm::vec4 vec4;

	typedef glm::ivec2 ivec2;
	typedef glm::ivec3 ivec3;
	typedef glm::ivec4 ivec4;

	typedef glm::mat2 mat2;
	typedef glm::mat3 mat3;
	typedef glm::mat4 mat4;

	typedef glm::quat quat;

	template <typename T> 
	struct datatype_traits 
	{
		typedef T type;
		typedef T base_type;
		static const size_t size = 1;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec2<T> > 
	{
		typedef glm::detail::tvec2<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 2;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec3<T> > 
	{
		typedef glm::detail::tvec3<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 3;
	};

	template <typename T> 
	struct datatype_traits< glm::detail::tvec4<T> > 
	{
		typedef glm::detail::tvec4<T> type;
		typedef typename type::value_type base_type;
		static const size_t size = 4;
	};

	using glm::max;
	using glm::min;

	enum datatype
	{
		NONE,
		INT,
		BYTE,
		SHORT,
		UINT,
		UBYTE,
		USHORT,
		FLOAT,
		FLOAT_VECTOR_2,
		FLOAT_VECTOR_3,
		FLOAT_VECTOR_4,
		FLOAT_MATRIX_2,
		FLOAT_MATRIX_3,
		FLOAT_MATRIX_4,
		INT_VECTOR_2,
		INT_VECTOR_3,
		INT_VECTOR_4,
		// unsupported gl conversion, remove?
		BYTE_VECTOR_2, 
		BYTE_VECTOR_3,
		BYTE_VECTOR_4,
		QUAT,
		TRANSFORM,
		DATATYPE_COUNT,
	};

	struct datatype_info
	{
		size_t   size;
		datatype base;
		size_t   elements;
	};

	inline const datatype_info& get_datatype_info( datatype dt )
	{
		static datatype_info info[ DATATYPE_COUNT ] = {
			{ 0, NONE, 0 },   // NONE  
			{ 4, INT, 1 },    // INT,
			{ 1, BYTE, 1 },   // BYTE,
			{ 2, SHORT, 1 },  // SHORT,
			{ 4, UINT, 1 },   // UINT,
			{ 1, UBYTE, 1 },  // UBYTE,
			{ 2, USHORT, 1 }, // USHORT,
			{ 4, FLOAT, 1 },  // FLOAT
			{ 4 * 2,  FLOAT, 2 },  // FLOAT_VECTOR_2,
			{ 4 * 3,  FLOAT, 3 },  // FLOAT_VECTOR_3,
			{ 4 * 4,  FLOAT, 4 },  // FLOAT_VECTOR_4,
			{ 4 * 4,  FLOAT, 4 },  // FLOAT_MATRIX_2,
			{ 4 * 9,  FLOAT, 9 },  // FLOAT_MATRIX_3,
			{ 4 * 16, FLOAT, 16 }, // FLOAT_MATRIX_4,
			{ 4 * 2,  INT, 2 },  // INT_VECTOR_2,
			{ 4 * 3,  INT, 3 },  // INT_VECTOR_3,
			{ 4 * 4,  INT, 4 },  // INT_VECTOR_4,
			// unsupported gl conversion, remove?
			{ 1 * 2,  BYTE, 2 },  // BYTE_VECTOR_2,
			{ 1 * 3,  BYTE, 3 },  // BYTE_VECTOR_3,
			{ 1 * 4,  BYTE, 4 },  // BYTE_VECTOR_4,
			{ 4 * 4,  FLOAT, 4 },      // QUAT,
			{ 7 * 4,  FLOAT, 7 },      // TRANSFORM,
		};
		return info[dt];
	}

	template < datatype EnumType > struct enum_to_type {};

	template <> struct enum_to_type< NONE >  { typedef void type; };
	template <> struct enum_to_type< INT >   { typedef int type; };
	template <> struct enum_to_type< UINT >  { typedef unsigned int type; };
	template <> struct enum_to_type< SHORT > { typedef short type; };
	template <> struct enum_to_type< USHORT >{ typedef unsigned short type; };
	template <> struct enum_to_type< BYTE >  { typedef char type; };
	template <> struct enum_to_type< UBYTE > { typedef unsigned char type; };
	template <> struct enum_to_type< FLOAT > { typedef f32 type; };

	template <> struct enum_to_type< FLOAT_VECTOR_2 > { typedef vec2 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_3 > { typedef vec3 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_4 > { typedef vec4 type; };

	template <> struct enum_to_type< INT_VECTOR_2 > { typedef ivec2 type; };
	template <> struct enum_to_type< INT_VECTOR_3 > { typedef ivec3 type; };
	template <> struct enum_to_type< INT_VECTOR_4 > { typedef ivec4 type; };

	template <> struct enum_to_type< BYTE_VECTOR_2 > { typedef i8vec2 type; };
	template <> struct enum_to_type< BYTE_VECTOR_3 > { typedef i8vec3 type; };
	template <> struct enum_to_type< BYTE_VECTOR_4 > { typedef i8vec4 type; };

	template <> struct enum_to_type< FLOAT_MATRIX_2 > { typedef mat2 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_3 > { typedef mat3 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_4 > { typedef mat4 type; };

	template <> struct enum_to_type< QUAT > { typedef quat type; };

	template < typename TYPE > struct type_to_enum {};

	template <> struct type_to_enum< long >          { static const datatype type = INT; };
	template <> struct type_to_enum< unsigned long > { static const datatype type = UINT; };
	template <> struct type_to_enum< int >           { static const datatype type = INT; };
	template <> struct type_to_enum< unsigned int >  { static const datatype type = UINT; };
	template <> struct type_to_enum< short >         { static const datatype type = SHORT; };
	template <> struct type_to_enum< unsigned short >{ static const datatype type = USHORT; };
	template <> struct type_to_enum< char >          { static const datatype type = BYTE; };
	template <> struct type_to_enum< signed char >   { static const datatype type = BYTE; };
	template <> struct type_to_enum< unsigned char > { static const datatype type = UBYTE; };
	template <> struct type_to_enum< f32 > { static const datatype type = FLOAT; };

	template <> struct type_to_enum< vec2 > { static const datatype type = FLOAT_VECTOR_2; };
	template <> struct type_to_enum< vec3 > { static const datatype type = FLOAT_VECTOR_3; };
	template <> struct type_to_enum< vec4 > { static const datatype type = FLOAT_VECTOR_4; };

	template <> struct type_to_enum< ivec2 > { static const datatype type = INT_VECTOR_2; };
	template <> struct type_to_enum< ivec3 > { static const datatype type = INT_VECTOR_3; };
	template <> struct type_to_enum< ivec4 > { static const datatype type = INT_VECTOR_4; };

	template <> struct type_to_enum< i8vec2 > { static const datatype type = BYTE_VECTOR_2; };
	template <> struct type_to_enum< i8vec3 > { static const datatype type = BYTE_VECTOR_3; };
	template <> struct type_to_enum< i8vec4 > { static const datatype type = BYTE_VECTOR_4; };

	template <> struct type_to_enum< mat2 > { static const datatype type = FLOAT_MATRIX_2; };
	template <> struct type_to_enum< mat3 > { static const datatype type = FLOAT_MATRIX_3; };
	template <> struct type_to_enum< mat4 > { static const datatype type = FLOAT_MATRIX_4; };
	template <> struct type_to_enum< quat > { static const datatype type = QUAT; };

	template < typename T >
	struct sizeof_type 
	{
		static const int result = sizeof( T );
	};

	template <>
	struct sizeof_type< void >
	{
		static const int result = 0;
	};

	template < datatype T >
	struct sizeof_enum_type 
	{
		static const int result = sizeof_type< typename enum_to_type<T>::type >::result;
	};

	template < typename T >
	T interpolate( const T& lhs, const T& rhs, float f )
	{
		return glm::mix( lhs, rhs, f );
	}

	template <>
	inline quat interpolate( const quat& lhs, const quat& rhs, float f )
	{
		return glm::slerp( lhs, rhs, f );
	}

	template <typename T>
	glm::detail::tvec3<T> normalize_safe( 
		const glm::detail::tvec3<T>& x, 
		const glm::detail::tvec3<T>& def = vec3()
	)
	{
		typename glm::detail::tvec3<T>::value_type sqr = x.x * x.x + x.y * x.y + x.z * x.z;
		return ( sqr > 0 ? x * glm::inversesqrt(sqr) : def );
	}



} // namespace nv

#endif // NV_MATH_HH
