#ifndef NV_GFX_PARTICLE_ENGINE
#define NV_GFX_PARTICLE_ENGINE

#include <nv/common.hh>
#include <nv/math.hh>
#include <nv/array.hh>
#include <nv/handle.hh>
#include <nv/lua/lua_state.hh>
#include <nv/gfx/texture_atlas.hh>
#include <nv/interface/scene_node.hh>
#include <nv/interface/context.hh>

namespace nv
{
	static const int MAX_PARTICLE_EMMITERS = 8;

	enum class particle_orientation
	{
		POINT,
		ORIENTED,
		ORIENTED_COMMON,
		PERPENDICULAR,
		PERPENDICULAR_COMMON,
	};
	enum class particle_origin
	{
		CENTER,
		TOP_LEFT,
		TOP_CENTER,
		TOP_RIGHT,
		CENTER_LEFT,
		CENTER_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_CENTER,
		BOTTOM_RIGHT,
	};
	enum class particle_emmiter_type
	{
		POINT,
		BOX,
		CYLINDER,
		SPHERE,
		CYLINDROID,
		ELLIPSOID,
		HOLLOW_CYLINDER,
		HOLLOW_SPHERE,
		HOLLOW_CYLINDROID,
		HOLLOW_ELLIPSOID,
	};

	struct particle_vtx
	{
		vec3 position;
		vec2 texcoord;
		vec4 color;
	};

	struct particle_quad
	{
		particle_vtx data[6];
	};

	struct particle
	{
		vec3   position;
		vec4   color;
		vec3   dir;
		float  velocity;
		vec2   size;
		//float  rotation;
		uint32 death;
	};


	struct particle_emmiter_data
	{
		particle_emmiter_type type;
		vec3   position;
		vec3   extents;
		vec3   hextents;
		vec3   iextents;
		vec3   ihextents;
		vec4   color_min;
		vec4   color_max;
		vec2   size_min;
		vec2   size_max;
		bool   precise;
		bool   square;
		float  angle;
		float  velocity_min;
		float  velocity_max;
		uint32 lifetime_min;
		uint32 lifetime_max;
		vec3   dir;
		vec3   odir;
		vec3   cdir;
		uint32 duration_min;
		uint32 duration_max;
		uint32 repeat_min;
		uint32 repeat_max;
		float  rate;
	};

	struct particle_emmiter_info
	{
		bool   active;
		uint32 next_toggle;
		float  last_create;
	};

	struct particle_system_data
	{
		vec3   gravity;
		uint32 quota;
		bool   local;
		vec3   common_up;
		vec3   common_dir;
		bool   accurate_facing;
		particle_orientation  orientation;
		particle_origin       origin;
		uint32                emmiter_count;
		particle_emmiter_data emmiters[MAX_PARTICLE_EMMITERS]; 
	};


	struct particle_system_info
	{
		bool test;
		uint32                last_update;
		particle*             particles;
		particle_quad*        quads;
		particle_emmiter_info emmiters[MAX_PARTICLE_EMMITERS];

		uint32                count;
		vertex_array          vtx_array;
		buffer                vtx_buffer;
		//		bool                           own_va;
		//		uint32                         offset;

		const particle_system_data*    data;
	};

	struct particle_system_tag {};
	typedef handle< uint32, 16, 16, particle_system_tag > particle_system;

	class particle_engine
	{
	public:
		particle_engine( context* a_context );
		void load( lua::table_guard& table );
		void draw( particle_system system, const render_state& rs, const scene_state& ss );
		particle_system create_system( const std::string& id );
		void release( particle_system system );
		void update( particle_system system, const scene_state& s, uint32 ms );
		void set_texcoords( particle_system system, vec2 a, vec2 b );
		~particle_engine();
	private:
		void generate_data( particle_system_info* info );
		void destroy_particles( particle_system_info* info, uint32 ms );
		void create_particles( particle_system_info* info, uint32 ms );
		void update_particles( particle_system_info* info, uint32 ms );
		void update_emmiters( particle_system_info* info, uint32 ms );

		device*                        m_device;
		context*                       m_context;

		program                        m_program_local;
		program                        m_program_world;

		mat4                           m_view_matrix;
		mat4                           m_model_matrix;
		vec3                           m_inv_view_dir;
		vec3                           m_camera_pos;

		entity_store< particle_system_info, particle_system > m_systems;
		std::unordered_map< std::string, uint32 >             m_names;
		std::vector< particle_system_data >                   m_data; // constant
	};

}

#endif // NV_GFX_PARTICLE_ENGINE
