// Copyright (C) 2012-2013 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gl/gl_device.hh"

#include "nv/gl/gl_window.hh"
#include "nv/gl/gl_program.hh"
#include "nv/logging.hh"
#include "nv/lib/sdl.hh"
#include "nv/lib/sdl_image.hh"
#include "nv/gl/gl_enum.hh"
#include "nv/lib/gl.hh"

using namespace nv;

window* gl_device::create_window( uint16 width, uint16 height, bool fullscreen )
{
	return new gl_window( this, width, height, fullscreen );
}

window* nv::gl_device::adopt_window( void* sys_w_handle, void* sys_dc )
{
	return new gl_window( this, sys_w_handle, sys_dc );
}



gl_device::gl_device()
{
	nv::load_sdl_library();
	m_info = NULL;

	if ( SDL_Init( SDL_INIT_VIDEO | SDL_INIT_JOYSTICK ) < 0 )
	{
		NV_LOG( LOG_CRITICAL, "Video initialization failed: " << SDL_GetError( ) );
		return; // TODO: Error report
	}

#if NV_SDL_VERSION == NV_SDL_12
	m_info = SDL_GetVideoInfo( );

	if ( !m_info )
	{
		NV_LOG( LOG_CRITICAL, "Video query failed: " << SDL_GetError( ) );
		return; // TODO: Error report
	}
#endif

}

program* gl_device::create_program( const string& vs_source, const string& fs_source )
{
	return new gl_program( vs_source, fs_source );
}

// this is a temporary function that will be removed once we find a way to 
// pass binary file data around
image_data* nv::gl_device::create_image_data( const std::string& filename )
{
	load_sdl_image_library();
	SDL_Surface* image = IMG_Load( filename.c_str() );
	if (!image)
	{
		NV_LOG( LOG_ERROR, "Image file " << filename.c_str() << " not found!" );
		return nullptr;
	}
	// TODO: BGR vs RGB, single channel
	assert( image->format->BytesPerPixel > 2 );
	image_format format(image->format->BytesPerPixel == 3 ? RGB : RGBA, UBYTE );
	image_data* data = new image_data( format, glm::ivec2( image->w, image->h ), (nv::uint8*)image->pixels );
	return data;
}

uint32 gl_device::get_ticks()
{
	return SDL_GetTicks();
}

void gl_device::delay( uint32 ms )
{
	return SDL_Delay( ms );
}

gl_device::~gl_device()
{
	while ( m_textures.size() > 0 )
		release( m_textures.get_handle(0) );
	while ( m_buffers.size() > 0 )
		release( m_buffers.get_handle(0) );

	SDL_Quit();
}

nv::texture nv::gl_device::create_texture( ivec2 size, image_format aformat, sampler asampler, void* data /*= nullptr */ )
{
	unsigned glid = 0;
	glGenTextures( 1, &glid );

	glBindTexture( GL_TEXTURE_2D, glid );

	// Detect if mipmapping was requested
	if (( asampler.filter_min != sampler::LINEAR && asampler.filter_min != sampler::NEAREST ) ||
		( asampler.filter_max != sampler::LINEAR && asampler.filter_max != sampler::NEAREST ))
	{
		glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
	}

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, (int)nv::sampler_filter_to_enum( asampler.filter_min ) );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, (int)nv::sampler_filter_to_enum( asampler.filter_max ) );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, (int)nv::sampler_wrap_to_enum( asampler.wrap_s) );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, (int)nv::sampler_wrap_to_enum( asampler.wrap_t) );

	if (data)
	{
		glTexImage2D( GL_TEXTURE_2D, 0, (GLint)nv::image_format_to_enum(aformat.format), size.x, size.y, 0, nv::image_format_to_enum(aformat.format), nv::datatype_to_gl_enum(aformat.type), data );
	}

	glBindTexture( GL_TEXTURE_2D, 0 );

	texture result = m_textures.create();
	gl_texture_info* info = m_textures.get( result );
	info->format  = aformat;
	info->sampler = asampler;
	info->size    = size;
	info->glid    = glid;
	return result;
}

void nv::gl_device::release( texture t )
{
	gl_texture_info* info = m_textures.get( t );
	if ( info )
	{
		if ( info->glid != 0 )
		{
			glDeleteTextures( 1, &(info->glid) );
		}
		m_textures.destroy( t );
	}
}

void nv::gl_device::release( buffer b )
{
	gl_buffer_info* info = m_buffers.get( b );
	if ( info )
	{
		if ( info->glid != 0 )
		{
			glDeleteBuffers( 1, &(info->glid) );
		}
		m_buffers.destroy( b );
	}
}

const texture_info* nv::gl_device::get_texture_info( texture t )
{
	return m_textures.get( t );
}

nv::buffer nv::gl_device::create_buffer( buffer_type type, buffer_hint hint, size_t size, const void* source /*= nullptr */ )
{
	unsigned glid   = 0;
	unsigned glenum = buffer_type_to_enum( type );
	glGenBuffers( 1, &glid );

	glBindBuffer( glenum, glid );
	glBufferData( glenum, (GLsizeiptr)size, source, buffer_hint_to_enum( hint ) );
	glBindBuffer( glenum, 0 );

	buffer result = m_buffers.create();
	gl_buffer_info* info = m_buffers.get( result );
	info->type = type;
	info->hint = hint;
	info->size = size;
	info->glid = glid;
	return result;
}

const buffer_info* nv::gl_device::get_buffer_info( buffer t )
{
	return m_buffers.get( t );
}
