// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_MESH_DATA_HH
#define NV_MESH_DATA_HH

#include <nv/common.hh>
#include <nv/math.hh>
#include <nv/string.hh>
#include <nv/array.hh>
#include <nv/gfx/animation.hh>
#include <nv/interface/vertex.hh>

namespace nv
{

	// TODO: friend mesh_data_creator class?
	// TODO: private const etc


	struct mesh_raw_channel
	{
		friend class mesh_creator;

		vertex_descriptor desc;
		uint8*            data;
		uint32            count;

		mesh_raw_channel() : data( nullptr ), count( 0 ) {}
		~mesh_raw_channel() 
		{
			if ( data != nullptr ) delete[] data;
		}

		// TODO: this could have more options if stored!
		buffer_type get_buffer_type() const
		{
			if ( count > 0 && desc.slots[0].vslot == slot::INDEX )
			{
				return INDEX_BUFFER;
			}
			return VERTEX_BUFFER;
		}

		uint32 size() const { return count * desc.size; }

		template < typename VTX >
		static mesh_raw_channel* create( uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc.initialize<VTX>();
			result->count = count;
			result->data  = (count > 0 ? ( new uint8[ result->size() ] ) : nullptr );
			return result;
		}
		static mesh_raw_channel* create( const vertex_descriptor& vtxdesc, uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc  = vtxdesc;
			result->count = count;
			result->data  = (count > 0 ? ( new uint8[ result->size() ] ) : nullptr );
			return result;
		}

		template < typename ITYPE >
		static mesh_raw_channel* create_index( uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc.initialize_index<ITYPE>();
			result->count = count;
			result->data  = (count > 0 ? ( new uint8[ result->size() ] ) : nullptr );
			return result;
		}

		// TODO: remove this
		static mesh_raw_channel* create_index( datatype etype, uint32 count = 0 )
		{
			mesh_raw_channel* result = new mesh_raw_channel();
			result->desc.initialize_index( etype );
			result->count = count;
			result->data  = (count > 0 ? ( new uint8[ result->size() ] ) : nullptr );
			return result;
		}

	};

	class mesh_data
	{
		friend class mesh_creator;
		friend class mesh_data_creator;
	public:
		explicit mesh_data() : m_index_channel( nullptr ) {}
		explicit mesh_data( const std::string& name ) : m_name(name), m_index_channel( nullptr ) {}

		void add_channel( mesh_raw_channel* channel ) 
		{
			NV_ASSERT( channel, "nullptr passed to add_channel!" );
			m_channels.push_back( channel );
			if ( channel->get_buffer_type() == INDEX_BUFFER )
			{
				NV_ASSERT( !m_index_channel, "second index channel!" );
				m_index_channel = channel;
			}
		}

		const std::vector< mesh_raw_channel* >& get_raw_channels() const { return m_channels; }
		const mesh_raw_channel*           get_index_channel() const { return m_index_channel; }
		size_t get_channel_count() const { return m_channels.size(); }
		const mesh_raw_channel* get_channel( size_t index ) const 
		{ 
			if ( m_channels.size() > index ) return m_channels[index];
			return nullptr;
		}

		const mesh_raw_channel* get_channel( slot s ) const 
		{ 
			for ( auto ch : m_channels )
			{
				for ( uint32 i = 0; i < ch->desc.count; ++i )
					if ( ch->desc.slots[i].vslot == s )
					{
						return ch;
					}
			}
			return nullptr;
		}

		int get_channel_index( slot s ) const 
		{ 
			for ( uint32 c = 0; c < m_channels.size(); ++c )
			{
				const mesh_raw_channel* ch = m_channels[c];
				for ( uint32 i = 0; i < ch->desc.count; ++i )
					if ( ch->desc.slots[i].vslot == s )
					{
						return c;
					}
			}
			return -1;
		}


		size_t get_count() const 
		{
			if ( m_index_channel ) return m_index_channel->count;
			if ( m_channels.size() > 0 ) return m_channels[0]->count;
			return 0;
		}

		size_t get_count( size_t channel ) const 
		{
			if ( m_channels.size() > channel ) return m_channels[channel]->count;
			return 0;
		}

		template < typename VTX >
		const mesh_raw_channel* get_channel() const
		{
			vertex_descriptor compare;
			compare.initialize<VTX>();
			for ( auto ch : m_channels )
			{
				if ( ch->desc == compare )
				{
					return ch;
				}
			}
			return nullptr;
		}

		template < typename VTX >
		const VTX* get_channel_data() const
		{
			vertex_descriptor compare;
			compare.initialize<VTX>();
			for ( auto ch : m_channels )
			{
				if ( ch->desc == compare )
				{
					return (VTX*)ch->data;
				}
			}
			return nullptr;
		}

		void set_name( const std::string& name ) { m_name = name; }
		const std::string& get_name() const { return m_name; }

		void move_to( mesh_data& other )
		{
			other.m_name          = m_name;
			other.m_index_channel = m_index_channel;
			for ( auto c : m_channels )
			{
				other.m_channels.push_back( c );
			}
			m_channels.clear();
			m_index_channel = nullptr;
		}

		virtual ~mesh_data()
		{
			for ( auto channel : m_channels ) delete channel;
		}
	private:
		std::string                      m_name;
		std::vector< mesh_raw_channel* > m_channels;
		mesh_raw_channel*                m_index_channel;
	};

	struct mesh_node_data
	{
		string    name;
		sint16    target_id;
		sint16    parent_id;
		mat4      transform;
		key_data* data;
	};

	class mesh_nodes_data
	{
		friend class mesh_creator;
		friend class mesh_nodes_creator;
	public:
		explicit mesh_nodes_data( const string& name, uint32 count, mesh_node_data* nodes )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(0), m_duration(0.0f), m_flat( false )
		{
		}

		explicit mesh_nodes_data( const string& name, uint32 count, mesh_node_data* nodes,
			uint16 a_fps, float a_frames, bool a_flat )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(a_fps), m_duration(a_frames), m_flat( a_flat )
		{
		}

		size_t get_count() const { return m_count; }

		const mesh_node_data* get_node( size_t i ) const 
		{
			if ( i >= m_count ) return nullptr;
			return &m_nodes[i];
		}

		const mesh_node_data* get_node( const std::string& name ) const 
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_nodes[ i ].name == name )
					return &m_nodes[ i ];
			}
			return nullptr;
		}

		int get_node_index( const std::string& name ) const 
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_nodes[ i ].name == name )
					return (int)i;
			}
			return -1;
		}

		key_data* release_node_data( size_t i )
		{
			key_data* result = m_nodes[i].data;
			m_nodes[i].data = nullptr;
			return result;
		}

		bool is_flat() const { return m_flat; }
		uint16 get_frame_rate() const { return m_frame_rate; }
		float get_duration() const { return m_duration; }
		void set_name( const std::string& name ) { m_name = name; }
		const string& get_name() const { return m_name; }

		~mesh_nodes_data()
		{
			if ( m_count > 0 )
				for ( uint32 i = 0; i < m_count; ++i ) delete m_nodes[i].data;
			delete[] m_nodes;
		}

	private:
		string m_name;
		uint32 m_count;
		mesh_node_data* m_nodes;
		uint16  m_frame_rate;
		float   m_duration;
		bool    m_flat;
	};

	class mesh_data_pack
	{
		friend class mesh_creator;
	public:
		explicit mesh_data_pack( uint32 a_count, mesh_data* a_meshes, mesh_nodes_data* a_nodes = nullptr )
		{
			m_count  = a_count;
			m_meshes = a_meshes;
			m_nodes  = a_nodes;
		}
		const mesh_data* get_mesh( uint32 index ) const
		{
			if ( index >= m_count ) return nullptr;
			return &m_meshes[ index ];
		}
		const mesh_data* get_mesh( const std::string& name ) const
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_meshes[ i ].get_name() == name )
					return &m_meshes[ i ];
			}
			return nullptr;
		}
		uint32 get_count() const { return m_count; }
		const mesh_nodes_data* get_nodes() const { return m_nodes; }
		~mesh_data_pack()
		{
			delete[] m_meshes;
			delete   m_nodes;
		}
	private:
		uint32           m_count;
		mesh_data*       m_meshes;
		mesh_nodes_data* m_nodes;
	};
}

#endif // NV_MESH_DATA_HH
