// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file context.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Context class
 */

#ifndef NV_CONTEXT_HH
#define NV_CONTEXT_HH

#include <nv/common.hh>
#include <nv/interface/device.hh>
#include <nv/interface/camera.hh>
#include <nv/interface/program.hh>
#include <nv/interface/vertex_buffer.hh>
#include <nv/interface/clear_state.hh>
#include <nv/interface/render_state.hh>

namespace nv
{
	enum primitive
	{
		POINTS,
		LINES,
		LINE_LOOP,
		LINE_STRIP,
		TRIANGLES,
		TRIANGLE_STRIP,
		TRIANGLE_FAN
	};

	class mesh_interface
	{
	public:
		mesh_interface() : m_va( nullptr ) {}
		explicit mesh_interface( vertex_array* array ) : m_va( array ) {}
		virtual void update( uint32 ) {}
		virtual void update( program* ) {}
		virtual nv::vertex_array* get_vertex_array() const { return m_va; }
		virtual size_t get_index_count() const { return 0; }
		virtual nv::primitive get_primitive_type() const { return nv::TRIANGLES; }
		virtual ~mesh_interface() {}
	protected:
		vertex_array* m_va;
	};

	class device;
	class texture2d;
	class context
	{
	public:
		context( device* a_device )	
		{ 
			// TODO: this will pose a problem in case of multiple contexts
			program::initialize_engine_uniforms();
			m_device = a_device; 
		}
		virtual void bind( texture, texture_slot ) = 0;
		virtual void bind( vertex_buffer* ) = 0;
		virtual void bind( index_buffer* ) = 0;
		virtual void bind( program* ) = 0;
		virtual void bind( vertex_array* ) = 0;
		virtual void unbind( vertex_buffer* ) = 0;
		virtual void unbind( index_buffer* ) = 0;
		virtual void unbind( program* ) = 0;
		virtual void unbind( vertex_array* ) = 0;
		virtual void update( texture, void* ) = 0;
		virtual void update( index_buffer*, const void*, size_t /*offset*/, size_t /*size*/ ) = 0;
		virtual void update( vertex_buffer*, const void*, size_t /*offset*/, size_t /*size*/ ) = 0;

		virtual void clear( const clear_state& cs ) = 0;
		// temporary
		virtual void draw( primitive prim, const render_state& rs, program* p, vertex_array* va, size_t count ) = 0;

		virtual void draw( const render_state& rs, program* p, const mesh_interface* mesh )
		{
			draw( mesh->get_primitive_type(), rs, p, mesh->get_vertex_array(), mesh->get_index_count() );
		}
		virtual void draw( const render_state& rs, const scene_state& s, program* p, const mesh_interface* mesh )
		{
			p->apply_engine_uniforms( this, &s );
			draw( rs, p, mesh );
		}
		virtual void draw( primitive prim, const render_state& rs, const scene_state& s, program* p, vertex_array* va, size_t count )
		{
			p->apply_engine_uniforms( this, &s );
			draw( prim, rs, p, va, count );
		}

		virtual void apply_render_state( const render_state& state ) = 0;
		virtual const ivec4& get_viewport() = 0;
		virtual void set_viewport( const ivec4& viewport ) = 0;
		virtual device* get_device() { return m_device; }
		virtual ~context() 
		{
			// TODO: this will pose a problem in case of multiple contexts
			program::destroy_engine_uniforms();
		}
	protected:
		device*      m_device;
		clear_state  m_clear_state;
		render_state m_render_state;
		ivec4        m_viewport;
	};

} // namespace nv

#endif // NV_CONTEXT_HH
