// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file device.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Device class
 */

#ifndef NV_DEVICE_HH
#define NV_DEVICE_HH

#include <nv/common.hh>
#include <nv/string.hh>
#include <nv/handle.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/interface/vertex_buffer.hh>
#include <nv/interface/image_data.hh>

namespace nv
{
	class window;
	class program;

	struct sampler
	{
		enum filter
		{
			LINEAR,
			NEAREST,
			NEAREST_MIPMAP_NEAREST,
			LINEAR_MIPMAP_NEAREST,
			NEAREST_MIPMAP_LINEAR,
			LINEAR_MIPMAP_LINEAR
		};
		enum wrap
		{
			CLAMP_TO_EDGE,
			CLAMP_TO_BORDER, 
			MIRRORED_REPEAT, 
			REPEAT
		};

		filter filter_min;
		filter filter_max;
		wrap wrap_s;
		wrap wrap_t;

		sampler() : filter_min( LINEAR ), filter_max( LINEAR ), wrap_s( REPEAT ), wrap_t( REPEAT ) {}
		sampler( filter min, filter max, wrap s, wrap t )
			: filter_min( min ), filter_max( max ), wrap_s( s ), wrap_t( t ) {}
		sampler( filter f, wrap w )
			: filter_min( f ), filter_max( f ), wrap_s( w ), wrap_t( w ) {}

	};

	struct texture_info
	{
		ivec2        size;
		image_format format;
		sampler      sampler;
	};


	struct texture_tag {};
	typedef handle< uint32, 16, 16, texture_tag > texture;

	class device
	{
	public:
		virtual window* create_window( uint16 width, uint16 height, bool fullscreen ) = 0;
		virtual window* adopt_window( void* sys_w_handle, void* sys_dc ) = 0;
		virtual program* create_program( const string& vs_source, const string& fs_source ) = 0;
		virtual vertex_buffer* create_vertex_buffer( buffer_hint hint, size_t size, const void* source = nullptr ) = 0;
		virtual index_buffer* create_index_buffer( buffer_hint hint, size_t size, const void* source = nullptr ) = 0;
		virtual vertex_array* create_vertex_array() = 0;
		virtual image_data* create_image_data( const std::string& filename ) = 0; // temporary
		virtual texture create_texture( ivec2 size, image_format aformat, sampler asampler, void* data = nullptr ) = 0;
		virtual void release_texture( texture ) = 0;
		virtual const texture_info* get_texture_info( texture ) = 0;
		virtual uint32 get_ticks() = 0;
		virtual void delay( uint32 ms ) = 0;

		virtual texture create_texture( image_data* data, sampler asampler ) 
		{
			return create_texture( data->get_size(), data->get_format(), asampler, (void*)data->get_data() );
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( vertex_array*, vertex_buffer*, const std::false_type& )
		{
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( vertex_array* va, vertex_buffer* vb, const std::true_type& )
		{
			typedef vertex_slot_info< VTX, SLOT > vinfo;
			typedef datatype_traits< typename vinfo::value_type > dt_traits;
			va->add_vertex_buffer( SLOT, vb, type_to_enum< dt_traits::base_type >::type, dt_traits::size, vinfo::offset, sizeof( VTX ), false );
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer( vertex_array* va, vertex_buffer* vb )
		{
			add_vertex_buffer_impl< VTX, SLOT >( va, vb, std::integral_constant< bool, vertex_has_slot< VTX, SLOT >::value >() );
		}


		template < typename VTX >
		vertex_array* create_vertex_array( const VTX* v, size_t count, buffer_hint hint )
		{
			// TODO: vb will not be owned or freed!
			vertex_array*  va = create_vertex_array();
			vertex_buffer* vb = create_vertex_buffer( hint, count * sizeof( VTX ), v );
			add_vertex_buffer< VTX, slot::POSITION >  ( va, vb );
			add_vertex_buffer< VTX, slot::TEXCOORD >  ( va, vb );
			add_vertex_buffer< VTX, slot::NORMAL   >  ( va, vb );
			add_vertex_buffer< VTX, slot::TANGENT >   ( va, vb );
			add_vertex_buffer< VTX, slot::BONEINDEX > ( va, vb );
			add_vertex_buffer< VTX, slot::BONEWEIGHT >( va, vb );
			add_vertex_buffer< VTX, slot::COLOR >     ( va, vb );
			return va;
		}

		template < typename VTX >
		vertex_array* create_vertex_array( const std::vector< VTX >& data, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), hint );
		}

		template < typename VTX, typename IDX >
		vertex_array* create_vertex_array( const VTX* v, size_t vcount, const IDX* i, size_t icount, buffer_hint hint )
		{
			vertex_array* va = create_vertex_array( v, vcount, hint );
			index_buffer* ib = create_index_buffer( hint, icount * sizeof( IDX ), i );
			va->set_index_buffer( ib, type_to_enum< IDX >::type, true );
			return va;
		}

		template < typename VTX, typename IDX >
		vertex_array* create_vertex_array( const std::vector< VTX >& data, const std::vector< IDX >& idata, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), idata.data(), idata.size(), hint );
		}

		// TODO: HINTS ARE DIFFERENT!
		vertex_array* create_vertex_array( const mesh_data* data, buffer_hint hint )
		{
			vertex_array*  va = create_vertex_array();
			const std::vector< mesh_raw_channel* >& channels = data->get_raw_channels();
			for ( uint32 ch = 0; ch < channels.size(); ++ch )
			{
				const mesh_raw_channel* channel = channels[ch];
				if ( channel->count > 0 )
				{
					if ( channel->is_index() )
					{
						index_buffer* ib = create_index_buffer( hint, channel->size(), channel->data );
						va->set_index_buffer( ib, channel->desc.slots[0].etype, true );
					}
					else
					{
						vertex_buffer* vb = create_vertex_buffer( hint, channel->size(), channel->data );
						va->add_vertex_buffers( vb, channel );
					}
				}
			}
			return va;
		}


		virtual ~device() {}
	};

} // namespace nv


#endif // NV_DEVICE_HH
