// Copyright (C) 2012-2013 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gl/gl_program.hh"

#include "nv/gl/gl_enum.hh"
#include "nv/logging.hh"
#include "nv/lib/gl.hh"

#include <glm/glm.hpp>
#include <glm/gtc/type_ptr.hpp>

using namespace nv;

gl_shader::gl_shader( uint32 sh_type ) 
	: shader_type( sh_type ), object_id(0)
{
	// no op
}

gl_shader::gl_shader( uint32 sh_type, const string& shader_code )
	: shader_type( sh_type ), object_id(0)
{
	compile( shader_code );
}

gl_shader::~gl_shader()
{
	free();
}

bool gl_shader::compile( const string& shader_code )
{
	generate();

	const char* pc = shader_code.c_str();

	glShaderSource( object_id,   1, &pc, 0 );
	glCompileShader( object_id );

	return validate();
}

void gl_shader::generate()
{
	if ( is_valid() )
	{
		free();
	}
	object_id = glCreateShader( shader_type );
}

void gl_shader::free()
{
	glDeleteShader( object_id );
	object_id = 0;
}

bool gl_shader::validate()
{
	const uint32 buffer_size = 1024;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int compile_ok = GL_FALSE;
	glGetShaderiv(object_id, GL_COMPILE_STATUS, &compile_ok);
	glGetShaderInfoLog( object_id, buffer_size, &length, buffer );

	if ( length > 0 )
	{
		if ( compile_ok == 0 )
		{
			NV_LOG( LOG_ERROR, "Shader #" << object_id << " error: " << buffer );
		}
		else
		{
			NV_LOG( LOG_INFO, "Shader #" << object_id << " compiled successfully: " << buffer );
		}
	}
	else
	{
		NV_LOG( LOG_INFO, "Shader #" << object_id << " compiled successfully." );
	}
	return compile_ok != 0;
}

gl_program::gl_program( const string& vertex_program, const string& fragment_program )
	: vertex_shader( GL_VERTEX_SHADER ), fragment_shader( GL_FRAGMENT_SHADER )
{
	glid = glCreateProgram();
	compile( vertex_program, fragment_program );
}

gl_program::~gl_program()
{
	if ( glid != 0 )
	{
		// Detach the shaders from the program
		glDetachShader( glid, vertex_shader.get_id() );
		glDetachShader( glid, fragment_shader.get_id() );
		glDeleteProgram( glid );
	}
}

bool gl_program::compile( const string& vertex_program, const string& fragment_program )
{
	if (!vertex_shader.compile( vertex_program )) { return false; }
	if (!fragment_shader.compile( fragment_program )) { return false; }

	glBindAttribLocation( glid, static_cast<GLuint>( slot::POSITION   ), "nv_position"  );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::TEXCOORD   ), "nv_texcoord"  );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::NORMAL     ), "nv_normal"    );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::COLOR      ), "nv_color"     );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::TANGENT    ), "nv_tangent"   );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::BONEINDEX  ), "nv_boneindex" );
	glBindAttribLocation( glid, static_cast<GLuint>( slot::BONEWEIGHT ), "nv_boneweight");

	glAttachShader( glid, fragment_shader.get_id() );
	glAttachShader( glid, vertex_shader.get_id() );
	glLinkProgram( glid );

	if (!validate())
	{
		return false;
	}
	load_attributes();
	load_uniforms();
	return true;
}

bool gl_program::is_valid() const
{
	return glid != 0;
}

void gl_program::load_attributes()
{
	int params;
	glGetProgramiv( glid, GL_ACTIVE_ATTRIBUTES, &params );

	for ( unsigned i = 0; i < (unsigned)params; ++i )
	{
		int attr_nlen;
		int attr_len;
		unsigned attr_type;
		char name_buffer[128];

		glGetActiveAttrib( glid, i, 128, &attr_nlen, &attr_len, &attr_type, name_buffer );

		string name( name_buffer, size_t(attr_nlen) );

		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int attr_loc = glGetAttribLocation( glid, name.c_str() );

		m_attribute_map[ name ] = new attribute( name, attr_loc, gl_enum_to_datatype( attr_type ), attr_len );
	}
}

void gl_program::load_uniforms()
{
	int params;
	glGetProgramiv( glid, GL_ACTIVE_UNIFORMS, &params );

	for ( unsigned i = 0; i < size_t(params); ++i )
	{
		int uni_nlen;
		int uni_len;
		unsigned uni_type;
		char name_buffer[128];

		glGetActiveUniform( glid, i, 128, &uni_nlen, &uni_len, &uni_type, name_buffer );

		string name( name_buffer, size_t(uni_nlen) );
		
		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int uni_loc = glGetUniformLocation( glid, name.c_str() );
		datatype utype = gl_enum_to_datatype( uni_type );
		
		// check for array
		string::size_type arrchar = name.find('[');
		if ( arrchar != string::npos )
		{
			name = name.substr( 0, arrchar );
		}

		uniform_base* u = create_uniform( utype, name, uni_loc, uni_len );
		NV_ASSERT( u, "Unknown uniform type!" );
		m_uniform_map[ name ] = u;
	}

	apply_link_engine_uniforms();
	bind_engine_uniforms();
}

void gl_program::update_uniforms()
{
	for ( uniform_map::iterator i = m_uniform_map.begin(); 	i != m_uniform_map.end(); ++i ) 
	{
		uniform_base* ubase = i->second;
		if ( ubase->is_dirty() )
		{
			int uloc = ubase->get_location();
			switch( ubase->get_type() )
			{
			case FLOAT          : glUniform1fv( uloc, ubase->get_length(), ((uniform< enum_to_type< FLOAT >::type >*)( ubase ))->get_value() ); break;
			case INT            : glUniform1iv( uloc, ubase->get_length(), ((uniform< enum_to_type< INT >::type >*)( ubase ))->get_value() ); break;
			case FLOAT_VECTOR_2 : glUniform2fv( uloc, ubase->get_length(), (GLfloat*)((uniform< enum_to_type< FLOAT_VECTOR_2 >::type >*)( ubase ))->get_value()); break;
			case FLOAT_VECTOR_3 : glUniform3fv( uloc, ubase->get_length(), (GLfloat*)((uniform< enum_to_type< FLOAT_VECTOR_3 >::type >*)( ubase ))->get_value()); break;
			case FLOAT_VECTOR_4 : glUniform4fv( uloc, ubase->get_length(), (GLfloat*)((uniform< enum_to_type< FLOAT_VECTOR_4 >::type >*)( ubase ))->get_value()); break;
			case INT_VECTOR_2   : glUniform2iv( uloc, ubase->get_length(), (GLint*)((uniform< enum_to_type< INT_VECTOR_2 >::type >*)( ubase ))->get_value()); break;
			case INT_VECTOR_3   : glUniform3iv( uloc, ubase->get_length(), (GLint*)((uniform< enum_to_type< INT_VECTOR_3 >::type >*)( ubase ))->get_value()); break;
			case INT_VECTOR_4   : glUniform4iv( uloc, ubase->get_length(), (GLint*)((uniform< enum_to_type< INT_VECTOR_4 >::type >*)( ubase ))->get_value()); break;
			case FLOAT_MATRIX_2 : glUniformMatrix2fv( uloc, ubase->get_length(), GL_FALSE, (GLfloat*)((uniform< enum_to_type< FLOAT_MATRIX_2 >::type >*)( ubase ))->get_value()); break;
			case FLOAT_MATRIX_3 : glUniformMatrix3fv( uloc, ubase->get_length(), GL_FALSE, (GLfloat*)((uniform< enum_to_type< FLOAT_MATRIX_3 >::type >*)( ubase ))->get_value()); break;
			case FLOAT_MATRIX_4 : glUniformMatrix4fv( uloc, ubase->get_length(), GL_FALSE, (GLfloat*)((uniform< enum_to_type< FLOAT_MATRIX_4 >::type >*)( ubase ))->get_value()); break;
			default : break; // error?
			}
			ubase->clean();
		}
	}

}


bool gl_program::validate()
{
	const uint32 buffer_size = 2048;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int status;

	glGetProgramiv( glid, GL_LINK_STATUS, &status );
	glGetProgramInfoLog( glid, buffer_size, &length, buffer );

	NV_LOG( LOG_INFO, "Program #" << glid << (status == GL_FALSE ? " failed to compile!" : " compiled successfully.") );

	if ( length > 0 )
	{
		NV_LOG( LOG_INFO, "Program #" << glid << " log: " << buffer );
	}

	if ( status == GL_FALSE ) 
	{
		return false;
	}

	glValidateProgram( glid );
	glGetProgramiv( glid, GL_VALIDATE_STATUS, &status );

	if ( status == GL_FALSE )
	{
		glGetProgramInfoLog( glid, buffer_size, &length, buffer );
		NV_LOG( LOG_ERROR, "Program #" << glid << " validation error : " << buffer );
		return false;
	}
	return true;
}


