// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gfx/mesh_creator.hh"

struct nv_key_transform { nv::transform tform; };

void nv::mesh_nodes_creator::pre_transform_keys()
{
	if ( m_data->m_flat ) return;
	merge_keys();
	uint32 max_frames = 0;
	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		sint16 parent_id = m_data->m_nodes[i].parent_id;
		key_data* keys   = m_data->m_nodes[i].data;
		key_data* pkeys  = ( parent_id != -1 ? m_data->m_nodes[parent_id].data : nullptr );
		size_t count     = ( keys ? keys->get_channel(0)->count : 0 );
		size_t pcount    = ( pkeys ? pkeys->get_channel(0)->count : 0 );
		max_frames = glm::max<uint32>( count, max_frames );
		if ( pkeys && pkeys->get_channel_count() > 0 && keys && keys->get_channel_count() > 0 )
		{
			nv_key_transform*  channel = ((nv_key_transform*)(keys->get_channel(0)->data));
			nv_key_transform* pchannel = ((nv_key_transform*)(pkeys->get_channel(0)->data));
			for ( unsigned n = 0; n < count; ++n )
			{
				channel[n].tform = pchannel[ glm::min( n, pcount-1 ) ].tform * channel[n].tform;
			}
		}
	}

	// DAE pre_transform hack
	if ( m_data->m_frame_rate == 1 )
	{
		m_data->m_frame_rate = 32;
		m_data->m_duration   = (float)max_frames;
	}

	m_data->m_flat = true;
}

// TODO: DELETE
struct assimp_key_p  { float time; nv::vec3 position; };
struct assimp_key_r  { float time; nv::quat rotation; };


void nv::mesh_nodes_creator::merge_keys()
{
	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		key_data* old_keys = m_data->m_nodes[i].data;
		if ( old_keys && old_keys->get_channel_count() > 0 )
		{
			size_t chan_count = old_keys->get_channel_count();
			if ( chan_count == 1 
				&& old_keys->get_channel(0)->desc.count == 1 
				&& old_keys->get_channel(0)->desc.slots[0].etype == TRANSFORM ) continue;

			size_t max_keys = 0;
			for ( size_t c = 0; c < chan_count; ++c )
			{
				max_keys = glm::max( max_keys, old_keys->get_channel(c)->count );
			}

			key_raw_channel* raw_channel = key_raw_channel::create<nv_key_transform>( max_keys );
			key_data* new_keys = new key_data;
			new_keys->add_channel( raw_channel );
			nv_key_transform* channel = ((nv_key_transform*)(raw_channel->data));
			key_descriptor final_key = old_keys->get_final_key();

			for ( unsigned n = 0; n < max_keys; ++n )
			{
				float key[ 16 ];
				float* pkey = key;

				for ( uint16 c = 0; c < chan_count; ++c )
				{
					size_t idx = glm::min( old_keys->get_channel(c)->count - 1, n );
					pkey += old_keys->get_channel(c)->get_raw( idx, pkey );
				}
				channel[n].tform = extract_transform_raw( final_key, key );
			}

			delete old_keys;
			m_data->m_nodes[i].data = new_keys;
		}
	}
}

void nv::mesh_nodes_creator::transform( float scale, const mat3& r33 )
{
	mat3 ri33 = glm::inverse( r33 );
	mat4 pre_transform ( scale * r33 );
	mat4 post_transform( 1.f/scale * ri33 ); 

	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		mesh_node_data& node = m_data->m_nodes[i];
		node.transform = pre_transform * node.transform * post_transform;
		if ( node.data )
		{
			key_data* kdata  = node.data;
			for ( size_t c = 0; c < kdata->get_channel_count(); ++c )
			{
				const key_raw_channel* channel = kdata->get_channel(c);
				size_t key_size = channel->desc.size;
				for ( size_t n = 0; n < channel->count; ++n )
				{
					transform_key_raw( channel->desc, (uint8*)(channel->data + n * key_size), scale, r33, ri33 );
				}
			}
		}
	}
}

void nv::mesh_data_creator::transform( float scale, const mat3& r33 )
{
	vec3 vertex_offset     = glm::vec3(); 
	mat3 vertex_transform  = scale * r33;
	mat3 normal_transform  = r33;

	for ( uint32 c = 0; c < m_data->get_channel_count(); ++c )
	{
		const mesh_raw_channel* channel = m_data->get_channel(c);
		const vertex_descriptor& desc   = channel->desc;
		uint8* raw_data = channel->data;
		int vtx_size = desc.size;
		int p_offset = -1;
		int n_offset = -1;
		int t_offset = -1;
		for ( uint32 i = 0; i < desc.count; ++i )
			switch ( desc.slots[i].vslot )
			{
				case slot::POSITION : if ( desc.slots[i].etype == FLOAT_VECTOR_3 ) p_offset = desc.slots[i].offset; break;
				case slot::NORMAL   : if ( desc.slots[i].etype == FLOAT_VECTOR_3 ) n_offset = desc.slots[i].offset; break;
				case slot::TANGENT  : if ( desc.slots[i].etype == FLOAT_VECTOR_4 ) t_offset = desc.slots[i].offset; break;
				default             : break;
			}

		if ( p_offset != -1 )
			for ( uint32 i = 0; i < channel->count; i++)
			{
				vec3& p = *((vec3*)(raw_data + vtx_size*i + p_offset ));
				p = vertex_transform * p + vertex_offset;
			}

		if ( n_offset != -1 )
			for ( uint32 i = 0; i < channel->count; i++)
			{
				vec3& n = *((vec3*)(raw_data + vtx_size*i + n_offset ));
				n = glm::normalize( normal_transform * n );
			}
		if ( t_offset != -1 )
			for ( uint32 i = 0; i < channel->count; i++)
			{
				vec4& t = *((vec4*)(raw_data + vtx_size*i + t_offset ));
				t = vec4( glm::normalize( normal_transform * vec3(t) ), t[3] );
			}
	}
}
