// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file mesh_loader.hh
 * @author Kornel Kisielewicz
 * @brief mesh loader
 */

#ifndef NV_MESH_LOADER_HH
#define NV_MESH_LOADER_HH

#include <nv/common.hh>
#include <vector>
#include <unordered_map>
#include <nv/transform.hh>
#include <nv/string.hh>
#include <nv/gfx/animation.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/interface/stream.hh>

namespace nv 
{

	struct mesh_node_data
	{
		string    name;
		sint16    target_id;
		sint16    parent_id;
		mat4      transform;
		key_data* data;
	};

	class mesh_nodes_data
	{
	public:
		explicit mesh_nodes_data( const string& name, uint32 count, mesh_node_data* nodes )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(0), m_duration(0.0f), m_flat( false )
		{
		}

		explicit mesh_nodes_data( const string& name, uint32 count, mesh_node_data* nodes,
			uint16 a_fps, float a_frames, bool a_flat )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(a_fps), m_duration(a_frames), m_flat( a_flat )
		{
		}

		size_t get_count() const { return m_count; }

		const mesh_node_data* get_node( size_t i ) const 
		{
			if ( i >= m_count ) return nullptr;
			return &m_nodes[i];
		}

		key_data* release_node_data( size_t i )
		{
			key_data* result = m_nodes[i].data;
			m_nodes[i].data = nullptr;
			return result;
		}

		bool is_flat() const { return m_flat; }
		uint16 get_frame_rate() const { return m_frame_rate; }
		float get_duration() const { return m_duration; }
		void set_name( const std::string& name ) { m_name = name; }
		const string& get_name() const { return m_name; }

		~mesh_nodes_data()
		{
			if ( m_count > 0 )
				for ( uint32 i = 0; i < m_count; ++i ) delete m_nodes[i].data;
			delete[] m_nodes;
		}

	private:
		string m_name;
		uint32 m_count;
		mesh_node_data* m_nodes;
		uint16  m_frame_rate;
		float   m_duration;
		bool    m_flat;
	};

	class mesh_data_pack
	{
	public:
		explicit mesh_data_pack( uint32 a_count, mesh_data* a_meshes, mesh_nodes_data* a_nodes = nullptr )
		{
			m_count  = a_count;
			m_meshes = a_meshes;
			m_nodes  = a_nodes;
		}
		const mesh_data* get_mesh( uint32 index ) const
		{
			if ( index >= m_count ) return nullptr;
			return &m_meshes[ index ];
		}
		uint32 get_count() const { return m_count; }
		const mesh_nodes_data* get_nodes() const { return m_nodes; }
		~mesh_data_pack()
		{
			delete[] m_meshes;
			delete   m_nodes;
		}
	private:
		uint32           m_count;
		mesh_data*       m_meshes;
		mesh_nodes_data* m_nodes;
	};


	class mesh_loader
	{
	public:
		mesh_loader() {}
		virtual ~mesh_loader() {}
		virtual bool load( stream& source ) = 0;
		virtual mesh_data* release_mesh_data( size_t index = 0 ) = 0;
		virtual mesh_data_pack* release_mesh_data_pack() = 0;
		virtual size_t get_mesh_count() const = 0;
		virtual size_t get_nodes_data_count() const { return 0; }
		virtual mesh_nodes_data* release_mesh_nodes_data( size_t = 0 ) { return nullptr; }
	};

}

#endif // NV_MESH_LOADER_HH

