// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/formats/md5_loader.hh"

#include <glm/gtc/constants.hpp>
#include "nv/logging.hh"
#include "nv/io/std_stream.hh"
#include <cstring>

using namespace nv;

// based on http://tfc.duke.free.fr/coding/md5-specs-en.html

static void next_line( std::istream& stream )
{
	stream.ignore( std::numeric_limits<std::streamsize>::max(), '\n' );
}

static inline void discard( std::istream& stream, const std::string& token )
{
//	stream.ignore( std::numeric_limits<std::streamsize>::max(), ' ' );
	std::string discarded;
	stream >> discarded;
	assert( discarded == token );
}


static void remove_quotes( std::string& str )
{
	size_t n;
	while ( ( n = str.find('\"') ) != std::string::npos ) str.erase(n,1);
}

static void unit_quat_w( glm::quat& quat )
{
	float t = 1.0f - ( quat.x * quat.x ) - ( quat.y * quat.y ) - ( quat.z * quat.z );
	quat.w = ( t < 0.0f ? 0.0f : -sqrtf(t) );
}

bool md5_loader::load( stream& source )
{
	std_stream sstream( &source );
	std::string command;
	dynamic_array< md5_weight > weights;
	dynamic_array< md5_weight_info > weight_info;

	sstream >> command;
	while ( !sstream.eof() )
	{
		if ( command == "MD5Version" )
		{
			sstream >> m_md5_version;
			assert( m_md5_version == 10 );
		}
		else if ( command == "commandline" )
		{
			next_line( sstream ); 
		}
		else if ( command == "numJoints" )
		{
			sstream >> m_num_joints;
			m_joints.resize( m_num_joints );
		}
		else if ( command == "numMeshes" )
		{
			sstream >> m_num_meshes;
			m_meshes.resize( m_num_meshes );
			m_num_meshes = 0;
		}
		else if ( command == "joints" )
		{
			discard( sstream, "{" );
			md5_joint joint;
			for ( size_t i = 0; i < m_num_joints; ++i )
			{
				int parent_id;
				sstream >> joint.name >> parent_id;
				discard( sstream, "(" );
				sstream >> joint.pos.x >> joint.pos.y >> joint.pos.z;
				discard( sstream, ")" );
				discard( sstream, "(" );
				sstream >> joint.orient.x >> joint.orient.y >> joint.orient.z;
				remove_quotes( joint.name );
				unit_quat_w( joint.orient );
				m_joints[i] = joint;
				next_line( sstream );
			}
			discard( sstream, "}" );
		}
		else if ( command == "mesh" )
		{
			mesh_data* mesh = new mesh_data("md5_mesh");

			int num_verts   = 0;
			int num_tris    = 0;
			int num_weights = 0;

			discard( sstream, "{" );
			sstream >> command;
			while ( command != "}" ) 
			{
				if ( command == "shader" )
				{
					std::string shader;
					sstream >> shader;
					remove_quotes( shader );
					next_line( sstream );
				}
				else if ( command == "numverts")
				{
					sstream >> num_verts; 

					md5_vtx_t* tdata = nullptr;
					{
						mesh_raw_channel* ch_pnt = mesh_raw_channel::create<md5_vtx_pnt>( num_verts );
						mesh_raw_channel* ch_t   = mesh_raw_channel::create<md5_vtx_t>( num_verts );
						mesh_raw_channel* ch_pntiw = mesh_raw_channel::create<md5_vtx_pntiw>( num_verts );
						tdata = (md5_vtx_t*)ch_t->data;
						mesh->add_channel( ch_pnt );
						mesh->add_channel( ch_t );
						// TODO: hack to prevent rendering
						ch_pntiw->count = 0;
						mesh->add_channel( ch_pntiw );
					}
					weight_info.resize( num_verts );

					next_line( sstream );
					std::string line;
					for ( int i = 0; i < num_verts; ++i )
					{
						size_t weight_count;
						size_t start_weight;
						vec2 texcoord;

						std::getline( sstream, line );
						sscanf( line.c_str(), "%*s %*u ( %f %f ) %u %u", &(texcoord.x), &(texcoord.y), &(start_weight), &(weight_count) );
						weight_info[i].start_weight = start_weight;
						weight_info[i].weight_count = weight_count;
						tdata[i].texcoord = texcoord;
					}  
				}
				else if ( command == "numtris" )
				{
					sstream >> num_tris;

					mesh_raw_channel* ch_i = mesh_raw_channel::create_index<uint32>( num_tris * 3 );
					uint32* vtx_i                = (uint32*)ch_i->data;
					uint32 idx = 0;
					mesh->add_channel( ch_i );

					next_line( sstream );
					std::string line;
					for ( int i = 0; i < num_tris; ++i )
					{
						size_t ti0;
						size_t ti1;
						size_t ti2;

						std::getline( sstream, line );
						sscanf( line.c_str(), "%*s %*u %u %u %u )", &(ti0), &(ti1), &(ti2));

						vtx_i[idx++] = (uint32)ti0;
						vtx_i[idx++] = (uint32)ti1;
						vtx_i[idx++] = (uint32)ti2;
					}              
				}
				else if ( command == "numweights" )
				{
					sstream >> num_weights;
					weights.resize( num_weights );
					next_line( sstream );
					std::string line;
					for ( int i = 0; i < num_weights; ++i )
					{
						md5_weight weight;

						std::getline( sstream, line );
						sscanf( line.c_str(), "%*s %*u %u %f ( %f %f %f )", &(weight.joint_id), &(weight.bias), &(weight.pos.x), &(weight.pos.y), &(weight.pos.z));
 						weights[i] = weight;
					}
				}
				else
				{
					next_line( sstream );
				}

				sstream >> command;
			}

			prepare_mesh( weight_info.size(), mesh, weights.data(), weight_info.data() );

			m_meshes[ m_num_meshes ] = mesh;
			m_num_meshes++;
		}
		sstream >> command;
	}

	return true;
}

bool md5_loader::prepare_mesh( uint32 vtx_count, mesh_data* mdata, md5_weight* weights, md5_weight_info* weight_info )
{
	md5_vtx_pnt* vtcs = (md5_vtx_pnt*)mdata->get_channel< md5_vtx_pnt >()->data; 
	md5_vtx_pntiw* vtx_data = (md5_vtx_pntiw*)mdata->get_channel< md5_vtx_pntiw >()->data; 

	for ( uint32 i = 0; i < vtx_count; ++i )
	{
		size_t start_weight = weight_info[i].start_weight;
		size_t weight_count = weight_info[i].weight_count;
		md5_vtx_pntiw& vdata = vtx_data[i];
		md5_vtx_pnt& vtc = vtcs[i];

		vtc.position = glm::vec3(0);
		vtc.normal   = glm::vec3(0);
		vtc.tangent  = glm::vec3(0);

		std::sort( weights + start_weight, weights + start_weight + weight_count, [](const md5_weight& a, const md5_weight& b) -> bool { return a.bias > b.bias; } );

		if ( weight_count > 4 )
		{
			float sum = 0.0f;
			for ( size_t j = 0; j < 4; ++j )
			{
				sum += weights[start_weight + j].bias;
			}
			float ratio = 1.0f / sum;
			for ( size_t j = 0; j < 4; ++j )
			{
				weights[start_weight + j].bias = ratio * weights[start_weight + j].bias;
			}
			weight_count = 4;
		}

		for ( size_t j = 0; j < 4; ++j )
		{
			if ( j < weight_count )
			{
				vdata.boneindex[j]  = weights[start_weight + j].joint_id;
				vdata.boneweight[j] = weights[start_weight + j].bias;
			}
			else
			{
				vdata.boneindex[j]  = 0;
				vdata.boneweight[j] = 0.0f;
			}
		}

		for ( size_t j = 0; j < 4; ++j )
		{
			if ( j < weight_count )
			{
				md5_weight& weight = weights[start_weight + j];
				md5_joint&  joint  = m_joints[weight.joint_id];
				glm::vec3 rot_pos = joint.orient * weight.pos;

				vtc.position += ( joint.pos + rot_pos ) * weight.bias;
			}
		}
	}

	const uint32*    idata = (uint32*)mdata->get_index_channel()->data;
	const md5_vtx_t* tdata = mdata->get_channel_data<md5_vtx_t>();

	// Prepare normals
	uint32 tri_count = mdata->get_count() / 3;
	for ( unsigned int i = 0; i < tri_count; ++i )
	{
		uint32 ti0 = idata[ i * 3 ];
		uint32 ti1 = idata[ i * 3 + 1 ];
		uint32 ti2 = idata[ i * 3 + 2 ];
 
		glm::vec3 v1 = vtcs[ ti0 ].position;
		glm::vec3 v2 = vtcs[ ti1 ].position;
		glm::vec3 v3 = vtcs[ ti2 ].position;
		glm::vec3 xyz1 = v3 - v1;
		glm::vec3 xyz2 = v2 - v1;

		glm::vec3 normal = glm::cross( xyz1, xyz2 );

		vtcs[ ti0 ].normal += normal;
		vtcs[ ti1 ].normal += normal;
		vtcs[ ti2 ].normal += normal;

		const vec2& w1 = tdata[ ti0 ].texcoord;
		const vec2& w2 = tdata[ ti1 ].texcoord;
		const vec2& w3 = tdata[ ti2 ].texcoord;

		vec2 st1 = w3 - w1;
		vec2 st2 = w2 - w1;

		float coef = 1.0f / (st1.x * st2.y - st2.x * st1.y);

		vec3 tangent = (( xyz1 * st2.y ) - ( xyz2 * st1.y )) * coef;

		vtcs[ ti0 ].tangent += tangent;
		vtcs[ ti1 ].tangent += tangent;
		vtcs[ ti2 ].tangent += tangent;
	}

	for ( size_t i = 0; i < vtx_count; ++i )
	{
		md5_vtx_pntiw& vdata = vtx_data[i];

		glm::vec3 normal  = glm::normalize( vtcs[i].normal );
		glm::vec3 tangent = glm::normalize( vtcs[i].tangent );
		vtcs[i].normal   = normal;
		vtcs[i].tangent  = tangent;

		vdata.position = vtcs[i].position;
		vdata.normal   = glm::vec3(0);
 		vdata.tangent  = glm::vec3(0);
 
 		for ( size_t j = 0; j < 4; ++j )
 		{
 			const md5_joint&  joint  = m_joints[vdata.boneindex[j]];
 			vdata.normal  += ( normal  * joint.orient ) * vdata.boneweight[j];
 			vdata.tangent += ( tangent * joint.orient ) * vdata.boneweight[j];
 		}
	}

	return true;
}


md5_animation::md5_animation()
	: m_md5_version( 0 )
	, m_num_frames( 0 )
	, m_num_joints( 0 )
	, m_frame_rate( 0 )
	, m_num_animated_components( 0 )
	, m_anim_duration( 0 )
	, m_frame_duration( 0 )
{

}

md5_animation::~md5_animation()
{

}

bool md5_animation::load_animation( stream& source )
{
	std::vector<md5_joint_info> joint_infos;
	std::vector<transform>      base_frames;
	m_num_frames = 0;

	std_stream sstream( &source );
	std::string command;

	sstream >> command;
	while ( !sstream.eof() )
	{
		if ( command == "MD5Version" )
		{
			sstream >> m_md5_version;
			assert( m_md5_version == 10 );
		}
		else if ( command == "commandline" )
		{
			next_line( sstream ); 
		}
		else if ( command == "numFrames" )
		{
			sstream >> m_num_frames;
			next_line( sstream ); 
		}
		else if ( command == "numJoints" )
		{
			sstream >> m_num_joints;
			m_joints.reserve( m_num_joints );
			next_line( sstream ); 
		}
		else if ( command == "frameRate" )
		{
			sstream >> m_frame_rate;
			next_line( sstream ); 
		}
		else if ( command == "numAnimatedComponents" )
		{
			sstream >> m_num_animated_components;
			next_line( sstream ); 
		}
		else if ( command == "hierarchy" )
		{
			discard( sstream, "{" );
			for ( size_t i = 0; i < m_num_joints; ++i )
			{
				md5_joint_info joint;
				sstream >> joint.name >> joint.parent_id >> joint.flags >> joint.start_index;
				remove_quotes( joint.name );
				joint_infos.push_back( joint );
				m_joints.emplace_back( joint.parent_id );
				next_line( sstream ); 
			}
			discard( sstream, "}" );
		}
		else if ( command == "bounds" )
		{
			discard( sstream, "{" );
			next_line( sstream ); 
			for ( size_t i = 0; i < m_num_frames; ++i ) 
			{
//  				vec3 min;
//  				vec3 max;
//  				discard( sstream, "(" );
//  				sstream >> min.x >> min.y >> min.z;
//  				discard( sstream, ")" );
//  				discard( sstream, "(" );
//  				sstream >> max.x >> max.y >> max.z;
// 				m_bounds.push_back( bound );
				next_line( sstream ); 
			}

			discard( sstream, "}" );
			next_line( sstream ); 
		}
		else if ( command == "baseframe" )
		{
			discard( sstream, "{" );
			next_line( sstream ); 

			for ( size_t i = 0; i < m_num_joints; ++i )
			{
				transform base_frame;
				vec3 pos;
				quat orient;
				discard( sstream, "(" );
				sstream >> pos.x >> pos.y >> pos.z;
				discard( sstream, ")" );
				discard( sstream, "(" );
				sstream >> orient.x >> orient.y >> orient.z;
				next_line( sstream ); 

				base_frames.emplace_back( pos, orient );
			}
			discard( sstream, "}" );
			next_line( sstream ); 
		}
		else if ( command == "frame" )
		{
			std::vector<float> frame;
			int frame_id;
			sstream >> frame_id;
			discard( sstream, "{" );
			next_line( sstream ); 

			frame.reserve( m_num_animated_components );
			char buf[50];
			for ( size_t i = 0; i < m_num_animated_components; ++i )
			{
				sstream >> buf;
				frame.push_back((float)atof(buf));
			}

			build_frame_skeleton( joint_infos, base_frames, frame );

			discard( sstream, "}" );
			next_line( sstream ); 
		}

		sstream >> command;
	} 


	m_frame_duration = 1.0f / (float)m_frame_rate;
	m_anim_duration = ( m_frame_duration * (float)m_num_frames );

	return true;
}


void nv::md5_animation::update_skeleton( transform* skeleton, float anim_time ) const
{
	anim_time = glm::clamp( anim_time, 0.0f, m_anim_duration );
	float frame_num = anim_time * (float)m_frame_rate;
	size_t frame0 = (size_t)floorf( frame_num );
	size_t frame1 = (size_t)ceilf( frame_num );
	frame0 = frame0 % m_num_frames;
	frame1 = frame1 % m_num_frames;

	float interpolation = fmodf( anim_time, m_frame_duration ) / m_frame_duration;

	for ( size_t i = 0; i < m_num_joints; ++i )
	{
		const std::vector< transform >& keys = m_joints[i].keys;
		skeleton[i] = interpolate( keys[frame0], keys[frame1], interpolation );
	}
}

void md5_animation::build_frame_skeleton( const std::vector<md5_joint_info>& joint_infos, const std::vector<transform>& base_frames, const std::vector<float>& frame_data )
{
	size_t index = m_joints[0].keys.size();
	for ( unsigned int i = 0; i < joint_infos.size(); ++i )
	{
		unsigned int j = 0;

		const md5_joint_info& jinfo = joint_infos[i];


		int parent_id = jinfo.parent_id;

		vec3 pos    = base_frames[i].get_position();
		quat orient = base_frames[i].get_orientation();
		if ( jinfo.flags & 1 )  pos.x    = frame_data[ jinfo.start_index + j++ ];
		if ( jinfo.flags & 2 )  pos.y    = frame_data[ jinfo.start_index + j++ ];
		if ( jinfo.flags & 4 )  pos.z    = frame_data[ jinfo.start_index + j++ ];
		if ( jinfo.flags & 8 )  orient.x = frame_data[ jinfo.start_index + j++ ];
		if ( jinfo.flags & 16 ) orient.y = frame_data[ jinfo.start_index + j++ ];
		if ( jinfo.flags & 32 )	orient.z = frame_data[ jinfo.start_index + j++ ];
		unit_quat_w( orient );

		if ( parent_id >= 0 ) // Has a parent joint
		{
			const std::vector< transform >& ptv = m_joints[ size_t( parent_id ) ].keys;
			transform ptr;
			if ( ptv.size() > index ) ptr = ptv[ index ];
			glm::vec3 rot_pos = ptr.get_orientation() * pos;

			pos    = ptr.get_position() + rot_pos;
			orient = ptr.get_orientation() * orient;

			orient = glm::normalize( orient );
		}

		m_joints[i].keys.push_back( transform( pos, orient ) );
	}
}

mesh_data* nv::md5_loader::release_mesh_data( size_t index )
{
	mesh_data* result = m_meshes[ index ];
	m_meshes[ index ] = nullptr;
	return result;
}

mesh_nodes_data* nv::md5_loader::release_mesh_nodes_data()
{
	mesh_node_data* nodes = new mesh_node_data[ m_num_joints ];
	for ( uint32 i = 0; i < m_num_joints; ++i )
	{
		mesh_node_data& node = nodes[i];
		node.name      = m_joints[i].name;
		node.target_id = -1;
		node.parent_id = -1;
		node.transform = transform( m_joints[i].pos, m_joints[i].orient ).inverse().extract();
		node.data      = nullptr;
	}
	return new mesh_nodes_data( "nodes", m_num_joints, nodes );
}

mesh_data_pack* nv::md5_loader::release_mesh_data_pack()
{
	uint32 size = m_meshes.size();
	mesh_data* meshes = new mesh_data[ size ];
	for ( uint32 i = 0; i < size; ++i )
	{
		m_meshes[i]->move_to( meshes[i] );
		delete m_meshes[i];
		m_meshes[i] = nullptr;
	}
	return new mesh_data_pack( size, meshes, release_mesh_nodes_data() );
}


nv::md5_loader::~md5_loader()
{
	for ( auto m : m_meshes ) { if (m) delete m; }
}

