// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gui/gui_environment.hh"

#include "nv/gui/gui_renderer.hh"


	/*

	TODO: parse a lua stylesheet as per Trac wiki

	IDEA: Store everything in std::unordered_maps, with lua_value's?

	A lua_value is a variant stores strings as const char* that deletes them on destructor?
	Question is that there might not be much gained on speed anyway, due to Lua's speed.
	Special function field allows delayed per parse execution?

	*/

#include "nv/gfx/sliced_buffer.hh"
#include "nv/gfx/texture_atlas.hh"

nv::gui::environment::environment( window* w, const std::string& shader_path )
	: m_renderer( nullptr ), m_window( w ), m_screen( nullptr )
{
	m_area.dim( dimension( w->get_width(), w->get_height() ) );
	m_screen = new element( m_area );
	m_renderer = new renderer( w, shader_path );
}

void nv::gui::environment::load_style( const std::string& filename )
{
	m_renderer->load_style( filename );
}

nv::gui::element* nv::gui::environment::create_element( element* parent, const rectangle& r )
{
	element* result = new element( r );
	if ( parent == nullptr ) parent = m_screen;
	parent->add_child( result );
	return result;
}

void nv::gui::environment::destroy_element( element* e )
{
	destroy_children( e );
	e->detach();
	delete e;
}

void nv::gui::environment::update( element* e, uint32 elapsed )
{
//	e->on_update( elapsed );
	if ( e->is_visible() )
	{
		for ( object* i : *e )
		{
			update( ((element*)i), elapsed );
		}
	}
	if ( e->is_dirty() || e->m_render_data == nullptr )
	{
		m_renderer->redraw( e, elapsed );
		e->set_dirty( false );
	}
}

void nv::gui::environment::draw( element* e )
{
	if ( e->is_visible() )
	{
//		e->on_draw();
		m_renderer->draw( e );
		for ( object* i : *e )
		{
			draw((element*)i);
		}
	}
}

void nv::gui::environment::update()
{
	update( m_screen, 0 );
}

void nv::gui::environment::draw()
{
	draw( m_screen );
	m_renderer->draw();
}

void nv::gui::environment::add_child( element* child )
{
	m_screen->add_child( child );
}

void nv::gui::environment::destroy_children( element* e )
{
	while ( !e->m_children.empty() )
	{
		destroy_element( (element*)e->m_children.front() );
	}
}


nv::gui::environment::~environment()
{
	destroy_element( m_screen );
	delete m_renderer;
}

bool nv::gui::environment::process_io_event( const io_event& )
{
	return false;
}

bool nv::gui::environment::process_io_event( element* e, const io_event& ev )
{
	return e->m_parent ? process_io_event( (element*)e->m_parent, ev ) : false;
}

nv::gui::element* nv::gui::environment::get_element( const position& p )
{
	return get_deepest_child( m_screen, p );
}

nv::gui::element* nv::gui::environment::get_deepest_child( element* e, const position& p )
{
	if ( !e->is_visible() ) return nullptr;

	element* result = nullptr;
	element::list::reverse_iterator it = e->m_children.rbegin();

	while ( it != e->m_children.rend() )
	{
		result = get_deepest_child( (element*)(*it), p );
		if ( result ) return result;
		++it;
	}

	if ( e->contains( p ) ) return e;
	return nullptr;
}

void nv::gui::environment::move_to_top( element* child )
{
	element* parent = (element*)child->m_parent;
	if ( parent )
	{
		auto it = std::find( parent->m_children.begin(), parent->m_children.end(), (object*)child );
		if ( it != parent->m_children.end() )
		{
			parent->m_children.erase( it );
			parent->m_children.push_back( child );
			parent->m_dirty = true;
		}	
	}
}

void nv::gui::environment::move_to_bottom( element* child )
{
	element* parent = (element*)child->m_parent;
	if ( parent )
	{
		auto it = std::find( parent->m_children.begin(), parent->m_children.end(), (object*)child );
		if ( it != parent->m_children.end() )
		{
			parent->m_children.erase( it );
			parent->m_children.push_front( child );
			parent->m_dirty = true;
		}
	}
}

void nv::gui::environment::set_relative( element* e, const rectangle& r )
{
	e->m_dirty    = true;
	e->m_relative = r;
	recalculate_absolute( e );
}

void nv::gui::environment::set_relative( element* e, const position& p )
{
	set_relative( e, rectangle( p, p + e->m_relative.get_size() ) );
}

void nv::gui::environment::recalculate_absolute( element* e )
{
	rectangle pabsolute;

	if ( e->m_parent )
	{
		pabsolute = ((element*)(e->m_parent))->m_absolute;
	}

	e->m_absolute = e->m_relative + pabsolute.ul;

	for ( object* o : e->m_children )
	{
		recalculate_absolute( ((element*)o) );
	}
}

