// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file md5_loader.hh
 * @author Kornel Kisielewicz
 * @brief md5 loader
 */

#ifndef NV_MD5_LOADER_HH
#define NV_MD5_LOADER_HH

#include <nv/common.hh>
#include <unordered_map>
#include <vector>
#include <nv/interface/mesh_loader.hh>

namespace nv 
{

	struct md5_vtx_pnt
	{
		vec3 position;
		vec3 normal;
		vec3 tangent;
	};

	struct md5_vtx_t
	{
		vec2 texcoord;
	};

	struct md5_vtx_data
	{
		vec3  position;
		vec3  normal;
		vec3  tangent;
		ivec4 boneindex;
		vec4  boneweight;
	};

	class md5_animation
	{
	protected:
		friend class md5_loader;
		friend class md5_mesh_data;
		struct md5_joint_info
		{
			std::string name;
			int         parent_id;
			int         flags;
			size_t      start_index;
		};

		struct md5_joint
		{
			int              parent;
			transform_vector keys;

			md5_joint( int a_parent, size_t reserve ) : parent( a_parent ) { keys.reserve( reserve ); }
		};

	public:
		md5_animation();
		virtual ~md5_animation();

		bool load_animation( stream& source );
		void update_skeleton( std::vector<transform>& skeleton, float anim_time ) const;
		
		size_t get_num_joints() const { return m_num_joints; }
		size_t get_frame_rate() const { return m_frame_rate; }
		size_t get_frame_count() const { return m_num_frames; }

	protected:
		std::vector<md5_joint>  m_joints;
		void build_frame_skeleton( const std::vector<md5_joint_info>& joint_info, const std::vector<transform>& base_frames, const std::vector<float>& frame_data );

	private:
		size_t m_md5_version;
		size_t m_num_frames;
		size_t m_num_joints;
		size_t m_frame_rate;
		size_t m_num_animated_components;
		float m_frame_duration;
		float m_anim_duration;
	};

	class md5_mesh_instance 
	{
		friend class md5_mesh_data;
	public:
		const void* data() const { return m_pntdata; }
		uint32 get_index_count() const { return m_indices; }
		size_t size() const { return m_size * sizeof( md5_vtx_pnt ); }
		void apply( const std::vector< transform >& skeleton );
		const md5_mesh_data* get_mesh_data() const { return m_data; }
		~md5_mesh_instance() { delete[] m_pntdata; }
	private:
		md5_mesh_instance( const md5_mesh_data* a_data );

		uint32                   m_size;
		uint32                   m_indices;
		md5_vtx_pnt*             m_pntdata;
		std::vector< transform > m_pos_offset;
		const md5_mesh_data*     m_data;
	};

	class md5_mesh_data : public mesh_data
	{
	public:
		friend class md5_loader;
		friend class md5_mesh_instance;
		md5_mesh_instance* spawn() const;
	private:
		uint32*                     m_idata;
		md5_vtx_t*                  m_tdata;
		md5_vtx_pnt*                m_pntdata;
		std::string                 m_shader;
		std::vector< transform >    m_bone_offset;
		std::vector< md5_vtx_data > m_vtx_data;
	};


	class md5_loader : public mesh_loader
	{
	public:
		md5_loader() {}
		virtual ~md5_loader();
		virtual bool load( stream& source );
		virtual mesh_data* release_mesh_data( size_t index = 0 );
		virtual size_t get_mesh_count() const { return m_meshes.size(); }
	protected:
		struct md5_weight
		{
			size_t    joint_id;
			float     bias;
			glm::vec3 pos;
		};

		struct md5_weight_info
		{
			size_t     start_weight;
			size_t     weight_count;
		};

		struct md5_joint
		{
			std::string name;
			vec3  pos;
			quat  orient;
		};
	protected:
		bool prepare_mesh( md5_mesh_data* mdata, std::vector< md5_weight >& weights, std::vector< md5_weight_info >& weight_info );
	protected:
		uint32 m_md5_version;
		uint32 m_num_joints;
		uint32 m_num_meshes;

		std::vector<md5_joint>      m_joints;
		std::vector<md5_mesh_data*> m_meshes;
	};

}

#endif // NV_MD5_LOADER_HH
