// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: decouple from the type database
// TODO: paths for function registration?


#ifndef NV_LUA_HH
#define NV_LUA_HH

#include <nv/common.hh>
#include <istream>
#include <map>
#include <nv/flags.hh>
#include <nv/types.hh>

#include <nv/lua/lua_path.hh>
#include <nv/lua/lua_values.hh>
#include <nv/lua/lua_dispatch.hh>
#include <string>

namespace nv
{
	namespace lua
	{
		const int ref_none  = -2;
		const int ref_nil   = -1;
		const int ret_multi = -1;

		typedef int reference;

		class state_wrapper
		{
		public:
			state_wrapper( lua_State* a_state, bool a_owner ) : m_state( a_state ), m_owner( a_owner ), m_global( true ) {}
			virtual ~state_wrapper();

			template < typename R >
			R call( const path& p )
			{
				if ( push_function( p, m_global ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1 >
			R call( const path& p, const T1& p1 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_value( m_state, p1 );
					if ( call_function( 1, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2 >
			R call( const path& p, const T1& p1, const T2& p2 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_values( m_state, p1, p2 );
					if ( call_function( 2, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_values( m_state, p1, p2, p3 );
					if ( call_function( 3, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3, typename T4 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3, const T4& p4 )
			{
				if ( push_function( p, m_global ) )
				{
					detail::push_value( m_state, p1, p2, p3, p4 );
					if ( call_function( 4, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}

			template< typename R >
			R get( const path& p )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R, typename T >
			R raw_get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			bool is_defined( const path& p ) { return is_defined( p, m_global ); }
			void register_native_function( lfunction f, const char* name );

			template < typename F, F f >
			void register_function( const char* name )
			{
				register_native_function( detail::function_wrapper< F, f >, name );
			}

			template < typename C, typename F, F f >
			void register_function( const char* name )
			{
				register_native_function( detail::object_method_wrapper< C, F, f >, name );
			}

		protected:
			bool is_defined( const path& p, bool global );
			bool push_function( const path& p, bool global );
			int call_function( int nargs, int nresults );
		private:
			void push_global_table();
			void pop_global_table();
			void call_get();
			void call_get_raw();
		protected:
			lua_State* m_state;
			bool m_owner;
			bool m_global;
		};

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class table_guard;
		class state : public state_wrapper
		{
			friend class stack_guard;
			friend class table_guard;

		public:
			explicit state( bool load_libs = false );
			explicit state( lua_State* state );
			virtual ~state();
			bool do_string( const std::string& code, const std::string& name, int rvalues = 0 );
			bool do_stream( std::istream& stream, const std::string& name );
			bool do_file( const std::string& filename );
			int get_stack_size();
			void log_stack();
			lua_State* get_raw();
			reference register_object( object * o );
			reference register_object( object * o, const char* lua_name );
			reference register_proto( object * o, const char* storage );

			template< typename TYPE >
			type_entry& register_type( const char* name )
			{
				m_type_database->create_type<TYPE>( name );
			}

			template< typename TYPE, int SIZE >
			void register_enum( const char* name, const char* prefix, type_enum (&init_enums)[SIZE] )
			{
				m_type_database->create_type<TYPE>( name ).enums( init_enums );
				register_enum_values( name, prefix );
			}
			void register_enum_values( const std::string& name, const std::string& prefix = std::string() );

			void store_metadata( object* o, const std::string& metaname, void* pointer );
			void unregister_object( object * o );
			void register_native_object_method( const char* lua_name, const char* name, lfunction f );
			template < typename F, F f >
			void register_object_method( const char* lua_name, const char* name )
			{
				register_native_object_method( lua_name, name, detail::object_method_wrapper< typename memfn_class_type<F>::type, F, f > );
			}
			operator lua_State*() { return m_state; }
			type_database* get_type_database() { return m_type_database; }
		private:
			int load_string( const std::string& code, const std::string& name );
			int load_stream( std::istream& stream, const std::string& name );
			int load_file( const std::string& filename );
			int do_current( const std::string& name, int rvalues = 0 );
			void deep_pointer_copy( int index, void* obj );
		private:
			type_database* m_type_database;
		};

		class table_guard : public state_wrapper
		{
		public:
			table_guard( state* lstate, const path& p, bool global = true );
			table_guard( const table_guard& parent, const path& p );
			virtual ~table_guard();
			size_t get_size();
			bool has_field( const std::string& element );
			std::string get_string( const std::string& element, const std::string& defval = "" );
			char get_char( const std::string& element, char defval = ' ' );
			int get_integer( const std::string& element, int defval = 0 );
			unsigned get_unsigned( const std::string& element, unsigned defval = 0 );
			double get_double( const std::string& element, double defval = 0.0 );
			bool get_boolean( const std::string& element, bool defval = false );

		private:
			int m_level;
		};



	} // namespace lua

} // namespace nv

#endif // NV_LUA_HH
