// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file md5_loader.hh
 * @author Kornel Kisielewicz
 * @brief md5 loader
 */

#ifndef NV_MD5_LOADER_HH
#define NV_MD5_LOADER_HH

#include <nv/common.hh>
#include <unordered_map>
#include <vector>
#include <nv/interface/mesh_loader.hh>

namespace nv 
{

	struct md5_vtx_pnt
	{
		vec3 position;
		vec3 normal;
		vec3 tangent;
	};

	struct md5_vtx_t
	{
		vec2 texcoord;
	};

	struct md5_vtx_data
	{
		glm::vec3 normal;
		glm::vec3 tangent;
		size_t    start_weight;
		size_t    weight_count;
	};

	struct md5_weight
	{
		size_t    joint_id;
		float     bias;
		glm::vec3 pos;
	};

	class md5_animation
	{
	protected:
		friend class md5_loader;
		friend class md5_mesh_data;
		struct md5_joint_info
		{
			std::string name;
			int         parent_id;
			int         flags;
			size_t      start_index;
		};
		typedef std::vector<md5_joint_info> md5_joint_info_list;

		struct md5_bound
		{
			glm::vec3 min;
			glm::vec3 max;
		};
		typedef std::vector<md5_bound> md5_bound_list;

		struct md5_base_frame
		{
			glm::vec3 pos;
			glm::quat orient;
		};
		typedef std::vector<md5_base_frame> md5_base_frame_list;

		struct md5_frame_data
		{
			int frame_id;
			std::vector<float> frame_data;
		};
		typedef std::vector<md5_frame_data> md5_frame_data_list;

		struct md5_skeleton_joint
		{
			md5_skeleton_joint() : parent(-1), pos(0) {}
			md5_skeleton_joint( const md5_base_frame& copy ) : pos( copy.pos ), orient( copy.orient ) {}

			int       parent;
			glm::vec3 pos;
			glm::quat orient;
		};
		typedef std::vector<md5_skeleton_joint> md5_skeleton_joint_list;

		struct md5_frame_skeleton
		{
			md5_skeleton_joint_list joints;
		};
		typedef std::vector<md5_frame_skeleton> md5_frame_skeleton_list;


	public:
		md5_animation();
		virtual ~md5_animation();

		bool load_animation( stream& source );
		void update( float delta_time );
		void reset_animation()
		{
			m_anim_time = 0;
		}

		const md5_frame_skeleton& get_skeleton() const
		{
			return m_animated_skeleton;
		}

		size_t get_num_joints() const
		{
			return m_num_joints;
		}

		size_t get_frame_rate() const
		{
			return m_frame_rate;
		}

		size_t get_frame_count() const
		{
			return m_num_frames;
		}


		const md5_joint_info& get_joint_info( uint32 index ) const
		{
			assert( index < m_joint_infos.size() );
			return m_joint_infos[index];
		}

	protected:

		md5_joint_info_list     m_joint_infos;
		md5_bound_list          m_bounds;
		md5_base_frame_list     m_base_frames;
		md5_frame_data_list     m_frames;
		md5_frame_skeleton_list m_skeletons;

		md5_frame_skeleton      m_animated_skeleton;

		void build_frame_skeleton( md5_frame_skeleton_list& skeletons, const md5_joint_info_list& joint_info, const md5_base_frame_list& base_frames, const md5_frame_data& frame_data );
		void interpolate_skeletons( md5_frame_skeleton& final_skeleton, const md5_frame_skeleton& skeleton0, const md5_frame_skeleton& skeleton1, float interpolate );

	private:
		size_t m_md5_version;
		size_t m_num_frames;
		size_t m_num_joints;
		size_t m_frame_rate;
		size_t m_num_animated_components;

		float m_anim_duration;
		float m_frame_duration;
		float m_anim_time;
	};

	class md5_mesh_data : public mesh_data
	{
	public:
		friend class md5_loader;

		void apply( const md5_animation& animation );
	private:
		uint32*                     m_idata;
		md5_vtx_t*                  m_tdata;
		md5_vtx_pnt*                m_pntdata;
		std::string                 m_shader;
		std::vector< md5_vtx_data > m_vtx_data;
		std::vector< md5_weight   > m_weights;
	};


	class md5_loader : public mesh_loader
	{
	public:
		md5_loader() {}
		virtual ~md5_loader();
		virtual bool load( stream& source );
		virtual mesh_data* release_mesh_data() { return release_mesh_data( 0 ); }
		virtual mesh_data* release_mesh_data( uint32 mesh );
		bool check_animation( const md5_animation& animation ) const;
	protected:

		struct md5_joint
		{
			std::string name;
			int   parent_id;
			vec3  pos;
			quat  orient;
		};
	protected:
		bool prepare_mesh( md5_mesh_data* mdata );
	protected:
		uint32 m_md5_version;
		uint32 m_num_joints;
		uint32 m_num_meshes;

		std::vector<md5_joint>      m_joints;
		std::vector<md5_mesh_data*> m_meshes;
	};

}

#endif // NV_MD5_LOADER_HH
