#include <nv/common.hh>
#include <iomanip>
#include <nv/gfx/keyframed_mesh.hh>
#include <nv/interface/vertex_buffer.hh>
#include <nv/gl/gl_device.hh>
#include <nv/gfx/image.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/interface/program.hh>
#include <nv/interface/texture2d.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/io/c_file_system.hh>
#include <nv/formats/md5_loader.hh>
#include <nv/profiler.hh>
#include <nv/logging.hh>
#include <nv/logger.hh>
#include <nv/math.hh>
#include <nv/time.hh>
#include <nv/string.hh>
#include <nv/gfx/skeletal_mesh.hh>
#include <glm/gtx/rotate_vector.hpp>
#include <glm/gtc/matrix_access.hpp>
#include <glm/gtx/matrix_interpolation.hpp>

class application
{
public:
	application();
	bool initialize();
	bool run();
	~application();
protected:
	void load_animation( const std::string& path );
protected:


	nv::device*       m_device;
	nv::window*       m_window;
	nv::texture2d*    m_diffuse;
	nv::texture2d*    m_specular;
	nv::texture2d*    m_normal;
	nv::clear_state   m_clear_state;
	nv::render_state  m_render_state;
	nv::scene_state   m_scene_state;

	nv::skeletal_mesh* m_mesh;
	nv::program*       m_program;
	nv::md5_mesh_data* m_mesh_data;
	nv::md5_animation* m_animation;

};

application::application()
{
	NV_PROFILE( "app_construct" );
	m_device = new nv::gl_device();
	m_window = m_device->create_window( 800, 600, false );
	m_animation = nullptr;

	nv::sampler sampler( nv::sampler::LINEAR, nv::sampler::REPEAT );
	nv::image_data* data = m_device->create_image_data( "data/qshambler_d.png" );
	m_diffuse  = m_device->create_texture2d( data->get_size(), nv::RGBA, nv::UBYTE, sampler, (void*)data->get_data() );
	delete data;
	data = m_device->create_image_data( "data/qshambler_s.png" );
	m_specular = m_device->create_texture2d( data->get_size(), nv::RGBA, nv::UBYTE, sampler, (void*)data->get_data() );
	delete data;

	data = m_device->create_image_data( "data/qshambler_local.png" );
	m_normal = m_device->create_texture2d( data->get_size(), nv::RGBA, nv::UBYTE, sampler, (void*)data->get_data() );
	delete data;

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_render_state.depth_test.enabled = true;
	m_render_state.culling.enabled    = true;
	m_render_state.culling.order      = nv::culling::CCW;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;
}

bool application::initialize()
{
	NV_PROFILE( "app_initialize" );
	m_program = m_device->create_program( nv::slurp( "md5.vert" ), nv::slurp( "md5.frag" ) );

	nv::md5_loader* loader = nullptr;
	{
		NV_PROFILE("loader->load");
		nv::c_file_system fs;
		nv::stream* mesh_file = fs.open( "data/qshambler.md5mesh" );
		loader = new nv::md5_loader();
		loader->load( *mesh_file );
		delete mesh_file;
	}

	{
		NV_PROFILE("create_mesh");
		m_mesh_data = (nv::md5_mesh_data*)loader->release_mesh_data();
		m_mesh = new nv::skeletal_mesh( m_window->get_context(), m_mesh_data );
		delete loader;
	}

	load_animation( "data/idle02.md5anim" );
	return true;
}

bool application::run()
{
	nv::profiler::pointer()->log_report(); 
	NV_PROFILE( "app_run" );
	int keypress = 0;

	glm::vec3 move( 0, 50.f, 0 );

	nv::uint32 ticks   = m_device->get_ticks();
	nv::uint32 last_ticks;
	nv::fps_counter_class< nv::system_us_timer > fps_counter;

	//m_mesh->setup_animation( 0, m_mesh->get_max_frames(), 2, true );

	while(!keypress) 
	{
		last_ticks = ticks;
		ticks      = m_device->get_ticks();
		nv::uint32 elapsed = ticks - last_ticks;
		m_mesh->update( elapsed );
		{
			NV_PROFILE( "clear" );
			m_window->get_context()->clear( m_clear_state );
		}

		{
			NV_PROFILE( "update_sh" );

			glm::vec3 source( 150.0f, 0.0f, 0.0f );
			glm::vec3 eye = glm::rotate( source, (ticks / 20.f), glm::vec3( 0.0,1.0,0.0 ) );

			m_scene_state.get_camera().set_lookat(eye + move, glm::vec3(0.0f, 0.0f, 0.0f) + move, glm::vec3(0.0, 1.0, 0.0));
			m_scene_state.get_camera().set_perspective(60.0f, 1.0f*800.0f/600.0f, 0.1f, 1000.0f);

			m_diffuse ->bind( 0 );
			m_specular->bind( 1 );
			m_normal  ->bind( 2 );
			m_program->set_opt_uniform( "light_position", glm::vec3(180.0, 180.0, 0) );
			m_program->set_opt_uniform( "light_diffuse",  glm::vec4(0.7,0.7,0.7,1.0) );
			m_program->set_opt_uniform( "light_specular", glm::vec4(1.0,1.0,1.0,1.0) );
		}

		{
			NV_PROFILE( "draw" );
			m_scene_state.set_model(nv::mat4(
				1.f,0.f,0.f,0.f,
				0.f,0.f,1.f,0.f,
				0.f,1.f,0.f,0.f,
				0.f,0.f,0.f,1.f
			) );

			m_window->get_context()->draw( m_render_state, m_scene_state, m_program, m_mesh );
		}

		{
			NV_PROFILE( "swap" );
			m_window->swap_buffers();
		}

		{
			NV_PROFILE( "pollevent" );
			nv::io_event event;
			while(m_window->poll_event(event)) 
			{      
				switch (event.type) 
				{
				case nv::EV_QUIT:
					keypress = 1;
					break;
				case nv::EV_KEY:
					if (event.key.pressed)
					{
						switch (event.key.code) 
						{
						case nv::KEY_ESCAPE : keypress = 1; break;
						case nv::KEY_F1 : nv::profiler::pointer()->log_report(); break;
						default: break;
						}
					}
					break;
				default: break;
				}
			}
		}
		fps_counter.tick();
	}
	return true;
}

void application::load_animation( const std::string& path )
{
	delete m_animation;
	m_animation = nullptr;
	NV_PROFILE("load_animation");
	nv::c_file_system fs;
	nv::stream* anim_file = fs.open( path.c_str() );

	if ( anim_file != nullptr )
	{
		m_animation = new nv::md5_animation();
		if ( !m_animation->load_animation(*anim_file) )
		{
			delete m_animation;
			m_animation = nullptr;
		}
		m_mesh->setup_animation( m_animation );
		delete anim_file;
	}
}


application::~application()
{
	delete m_program;
	delete m_mesh;
	delete m_diffuse;
	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

