// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file md5_loader.hh
 * @author Kornel Kisielewicz
 * @brief md5 loader
 */

#ifndef NV_MD5_LOADER_HH
#define NV_MD5_LOADER_HH

#include <unordered_map>
#include <vector>
#include <nv/common.hh>
#include <nv/interface/mesh_loader.hh>

namespace nv 
{

	class md5_animation
	{
	protected:
		friend class md5_loader;
		struct md5_joint_info
		{
			std::string name;
			int         parent_id;
			int         flags;
			int         start_index;
		};
		typedef std::vector<md5_joint_info> md5_joint_info_list;

		struct md5_bound
		{
			glm::vec3 min;
			glm::vec3 max;
		};
		typedef std::vector<md5_bound> md5_bound_list;

		struct md5_base_frame
		{
			glm::vec3 pos;
			glm::quat orient;
		};
		typedef std::vector<md5_base_frame> md5_base_frame_list;

		struct md5_frame_data
		{
			int frame_id;
			std::vector<float> frame_data;
		};
		typedef std::vector<md5_frame_data> md5_frame_data_list;

		struct md5_skeleton_joint
		{
			md5_skeleton_joint() : parent(-1), pos(0) {}
			md5_skeleton_joint( const md5_base_frame& copy ) : pos( copy.pos ), orient( copy.orient ) {}

			int       parent;
			glm::vec3 pos;
			glm::quat orient;
		};
		typedef std::vector<md5_skeleton_joint> md5_skeleton_joint_list;

		struct md5_frame_skeleton
		{
			md5_skeleton_joint_list joints;
		};
		typedef std::vector<md5_frame_skeleton> md5_frame_skeleton_list;


	public:
		md5_animation();
		virtual ~md5_animation();

		bool load_animation( stream& source );
		void update( float delta_time );

		const md5_frame_skeleton& get_skeleton() const
		{
			return m_animated_skeleton;
		}

		size_t get_num_joints() const
		{
			return m_num_joints;
		}

		const md5_joint_info& get_joint_info( uint32 index ) const
		{
			assert( index < m_joint_infos.size() );
			return m_joint_infos[index];
		}

	protected:

		md5_joint_info_list     m_joint_infos;
		md5_bound_list          m_bounds;
		md5_base_frame_list     m_base_frames;
		md5_frame_data_list     m_frames;
		md5_frame_skeleton_list m_skeletons;

		md5_frame_skeleton      m_animated_skeleton;

		void build_frame_skeleton( md5_frame_skeleton_list& skeletons, const md5_joint_info_list& joint_info, const md5_base_frame_list& base_frames, const md5_frame_data& frame_data );
		void interpolate_skeletons( md5_frame_skeleton& final_skeleton, const md5_frame_skeleton& skeleton0, const md5_frame_skeleton& skeleton1, float interpolate );

	private:
		size_t m_md5_version;
		size_t m_num_frames;
		size_t m_num_joints;
		size_t m_frame_rate;
		size_t m_num_animated_components;

		float m_anim_duration;
		float m_frame_duration;
		float m_anim_time;
	};


	class md5_loader : public mesh_loader
	{
	public:
		md5_loader() {}
		virtual ~md5_loader() {}
		virtual bool load( stream& source );
		virtual size_t get_size();
		virtual mesh* release_mesh();
		virtual mesh* get_frame( sint32 /*frame*/ ) { return nullptr; }
		bool check_animation( const md5_animation& animation ) const;
		void apply( const md5_animation& animation );
//		virtual const md3_tag* get_tag( const std::string& name ) const;
//		virtual mat4 get_tag( sint32 frame, const std::string& name ) const;
//		size_t get_max_frames() const;
//		void load_tag_names( std::vector< std::string >& tags );
//		void load_tags( std::vector<mat4>& t, const std::string& tag );
//		void load_positions( std::vector<vec3>& p, sint32 frame =-1 );
//		void load_normals( std::vector<vec3>& n, sint32 frame =-1 );
//		void load_texcoords( std::vector<vec2>& t );
//		void load_indicies( std::vector<uint16>& idx );
	protected:
		typedef std::vector<glm::vec3> md5_vec_buffer;
		typedef std::vector<glm::vec2> md5_tc_buffer;
		typedef std::vector<uint32> md5_index_buffer;

		struct md5_vertex
		{
			glm::vec3 position;
			glm::vec3 normal;
			glm::vec3 tangent;
			glm::vec2 texcoord;
			size_t    start_weight;
			size_t    weight_count;
		};

		typedef std::vector<md5_vertex> md5_vertex_list;

		struct md5_triangle
		{
			int indices[3];
		};
		typedef std::vector<md5_triangle> md5_triangle_list;

		struct md5_weight
		{
			int       joint_id;
			float     bias;
			glm::vec3 pos;
		};
		typedef std::vector<md5_weight> md5_weight_list;

		struct md5_joint
		{
			std::string name;
			int   parent_id;
			vec3  pos;
			quat  orient;
		};
		typedef std::vector<md5_joint> md5_joint_list;

		struct md5_mesh
		{
			std::string       shader;
			md5_vertex_list   verts;
			md5_triangle_list tris;
			md5_weight_list   weights;

			md5_vec_buffer    position_buffer; 
			md5_vec_buffer    normal_buffer;
			md5_vec_buffer    tangent_buffer;
			md5_tc_buffer     texcoord_buffer;
			md5_index_buffer  index_buffer;
		};
		typedef std::vector<md5_mesh> md5_mesh_list;
	public:
		const md5_vec_buffer&   get_positions( uint32 mesh_id ) const { return m_meshes[mesh_id].position_buffer; }
		const md5_vec_buffer&   get_normals  ( uint32 mesh_id ) const { return m_meshes[mesh_id].normal_buffer; }
		const md5_vec_buffer&   get_tangents ( uint32 mesh_id ) const { return m_meshes[mesh_id].tangent_buffer; }
		const md5_tc_buffer&    get_texcoords( uint32 mesh_id ) const { return m_meshes[mesh_id].texcoord_buffer; }
		const md5_index_buffer& get_indices  ( uint32 mesh_id ) const { return m_meshes[mesh_id].index_buffer; }
		uint32 get_vertex_count( uint32 mesh_id ) const { return m_meshes[mesh_id].position_buffer.size(); }
		uint32 get_index_count( uint32 mesh_id )  const { return m_meshes[mesh_id].index_buffer.size(); }
	protected:
		bool prepare_mesh( md5_mesh& mesh );
		bool prepare_normals( md5_mesh& mesh );
		bool prepare_animated_mesh( md5_mesh& mesh, const md5_animation::md5_frame_skeleton& skel );
	protected:
		uint32 m_md5_version;
		uint32 m_num_joints;
		uint32 m_num_meshes;

		md5_joint_list m_joints;
		md5_mesh_list  m_meshes;

		//bool           m_animation_present;
		//md5_animation m_animation;

		size_t m_size;
		glm::mat4x4     m_model_matrix;
	};

}

#endif // NV_MD5_LOADER_HH
