// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: the call() templates are copied over in table/state - can we get around that?

#ifndef NV_LUA_HH
#define NV_LUA_HH

#include <istream>
#include <map>
#include <nv/common.hh>
#include <nv/flags.hh>
#include <nv/lua/lua_path.hh>
#include <nv/lua/lua_values.hh>
#include <string>

struct lua_State;

namespace nv
{
	namespace lua
	{
		const int ref_none  = -2;
		const int ref_nil   = -1;
		const int ret_multi = -1;

		class state;
		typedef int reference;

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class table_guard;
		class state
		{
			friend class stack_guard;
			friend class table_guard;
		public:
			explicit state( bool load_libs = false );
			explicit state( lua_State* state );
			bool do_string( const std::string& code, const std::string& name, int rvalues = 0 );
			bool do_stream( std::istream& stream, const std::string& name );
			bool do_file( const std::string& filename );
			int get_stack_size();
			void log_stack();
			bool is_defined( const path& p, bool global = true );
			lua_State* get_raw();
			reference register_object( object * o );
			reference register_object( object * o, const char* lua_name );
			void unregister_object( object * o );
			void register_enum( type_database* db, const std::string& name, const std::string& prefix = std::string() );
			operator lua_State*() { return L; }
			~state();

			template < typename R >
			R call( const path& p )
			{
				if ( push_function( p ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1 >
			R call( const path& p, const T1& p1 )
			{
				if ( push_function( p ) )
				{
					detail::push_value( L, p1 );
					if ( call_function( 1, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2 >
			R call( const path& p, const T1& p1, const T2& p2 )
			{
				if ( push_function( p ) )
				{
					detail::push_values( L, p1, p2 );
					if ( call_function( 2, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3 )
			{
				if ( push_function( p ) )
				{
					detail::push_values( L, p1, p2, p3 );
					if ( call_function( 3, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3, typename T4 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3, const T4& p4 )
			{
				if ( push_function( p ) )
				{
					detail::push_value( L, p1, p2, p3, p4 );
					if ( call_function( 4, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}

		private:
			bool push_function( const path& p, bool global = true );
			int call_function( int nargs, int nresults );
			int load_string( const std::string& code, const std::string& name );
			int load_stream( std::istream& stream, const std::string& name );
			int load_file( const std::string& filename );
			int do_current( const std::string& name, int rvalues = 0 );
			bool push( const std::string& path, bool global = true );
			void deep_pointer_copy( int index, void* obj );
		private:
			bool m_owner;
			lua_State* L;
		};

		class table_guard
		{
		public:
			table_guard( state* lstate, const path& p, bool global = true );
			table_guard( const table_guard& parent, const path& p );
			size_t get_size();
			bool has_field( const std::string& element );
			std::string get_string( const std::string& element, const std::string& defval = "" );
			char get_char( const std::string& element, char defval = ' ' );
			int get_integer( const std::string& element, int defval = 0 );
			unsigned get_unsigned( const std::string& element, unsigned defval = 0 );
			double get_double( const std::string& element, double defval = 0.0 );
			bool get_boolean( const std::string& element, bool defval = false );
			bool is_defined( const path& p );

			template< typename R, typename T >
			R get( const T& key )
			{
				detail::push_value( L->L, key );
				call_get();
				return detail::pop_return_value<R>( L->L );
			}

			template< typename R, typename T >
			R raw_get( const T& key )
			{
				detail::push_value( L->L, key );
				call_get_raw();
				return detail::pop_return_value<R>( L->L );
			}

			template< uint32 SIZE, typename T >
			flags< SIZE, T > get_flags( const std::string& element )
			{
				flags< SIZE, T > result;
				get_raw_flags( element, result.data(), result.size() );
				return result;
			}

			template< uint32 SIZE, typename T >
			void load_flags( const std::string& element, flags< SIZE, T >& flags )
			{
				get_raw_flags( element, flags.data(), flags.size() );
			}

			template < typename R >
			R call( const path& p )
			{
				if ( L->push_function( p, false ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1 >
			R call( const path& p, const T1& p1 )
			{
				if ( L->push_function( p, false ) )
				{
					detail::push_value( L, p1 );
					if ( L->call_function( 1, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2 >
			R call( const path& p, const T1& p1, const T2& p2 )
			{
				if ( L->push_function( p, false ) )
				{
					detail::push_values( L, p1, p2 );
					if ( L->call_function( 2, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3 )
			{
				if ( L->push_function( p, false ) )
				{
					detail::push_values( L, p1, p2, p3 );
					if ( L->call_function( 3, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}
			template < typename R, typename T1, typename T2, typename T3, typename T4 >
			R call( const path& p, const T1& p1, const T2& p2, const T3& p3, const T4& p4 )
			{
				if ( L->push_function( p, false ) )
				{
					detail::push_value( L, p1, p2, p3, p4 );
					if ( L->call_function( 4, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( L );
					}
				}
				return R();
			}

		private:
			bool push_function( const path& p );
			int call_function( int nargs, int nresults );
			void call_get();
			void call_get_raw();
			void get_raw_flags( const std::string& element, uint8* data, uint32 count );

			state* L;
			stack_guard m_guard;
		};



	} // namespace lua

} // namespace nv

#endif // NV_LUA_HH
