// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LUA_VALUES_HH
#define NV_LUA_VALUES_HH

#include <type_traits>
#include <nv/common.hh>
#include <nv/string.hh>

struct lua_State;

namespace nv
{
	namespace lua
	{
		struct passer
		{
			virtual void push( lua_State *L ) const = 0;
		};

		struct returner_base
		{
			virtual void pop( lua_State *L ) = 0;
		};

		template < typename T >
		struct returner : public returner_base
		{
			operator T() { return value; }
			operator const T() const { return value; }
		protected:
			T value;
		};

		template < typename T >
		struct pass_traits
		{
			static void push( lua_State *L, const T& p ) { p.push( L ); }
			static void pop( lua_State *L, T& p ) { p.pop( L ); }
		};

		namespace detail
		{
			void push_value( lua_State *L, long n );
			void push_value( lua_State *L, double n );
			void push_value( lua_State *L, bool n );
			void push_value( lua_State *L, const std::string& s );
			void push_value( lua_State *L, const char* s );
			void push_value( lua_State *L, object* o );
			void push_value( lua_State *L, void* p );

			template< typename D >
			typename std::enable_if<std::is_base_of<object, D>::value>::type
				push_value( lua_State *L, D* o )
			{
				push_value( L, (object*)o );
			}

 			template < typename T >
 			void push_value( lua_State *L, const T& p )
 			{
 				pass_traits<T>::push( L, p );
 			}

			template < typename T1 >
			void push_values( lua_State *L, const T1& p1 ) { push_value( L, p1 ); }
			template < typename T1, typename T2 >
			void push_values( lua_State *L, const T1& p1, const T2& p2 ) { push_value( L, p1 ); push_value( L, p2 ); }
			template < typename T1, typename T2, typename T3 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); }
			template < typename T1, typename T2, typename T3, typename T4 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3, const T4& p4 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); push_value( L, p4 ); }
			template < typename T1, typename T2, typename T3, typename T4, typename T5 >
			void push_values( lua_State *L, const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5 ) { push_value( L, p1 ); push_value( L, p2 ); push_value( L, p3 ); push_value( L, p4 ); push_value( L, p5 ); }

			void pop_value( lua_State *L, long& n );
			void pop_value( lua_State *L, double& n );
			void pop_value( lua_State *L, bool& n );
			void pop_value( lua_State *L, std::string& s );
			void pop_value( lua_State *L, object*& o );
			void pop_value( lua_State *L, void*& p );
			template < typename T >
			inline void pop_value( lua_State *L, T& p )
			{
				pass_traits<T>::pop( L, p );
			}
			template < typename T >
			inline T pop_return_value( lua_State *L )
			{
				T ret;
				pass_traits<T>::pop( L, ret );
				return ret;
			}

			template <>
			inline void pop_return_value<void>( lua_State* )
			{
			}

			template < typename T >
			struct return_count
			{
				const static int value = 1;
			};

			template <>
			struct return_count<void>
			{
				const static int value = 0;
			};
		}

	} // namespace lua

} // namespace nv

#endif // NV_LUA_VALUES_HH
