// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_RANDOM_HH
#define NV_RANDOM_HH

#include <random>
#include <nv/common.hh>
#include <nv/math.hh>

namespace nv
{

	class random
	{
	public:
		typedef std::mt19937::result_type result_type;
		typedef std::mt19937::result_type seed_type;

		random( seed_type seed = 0 );
		seed_type randomize();
		void set_seed( seed_type seed = 0 );
		static random& get();
		result_type rand();
		sint32 srand( sint32 val );
		uint32 urand( uint32 val );
		f32 frand( f32 val );
		sint32 srange( sint32 min, sint32 max );
		uint32 urange( uint32 min, uint32 max );
		f32 frange( f32 min, f32 max );
		uint32 dice( uint32 count, uint32 sides );

		template < typename T >
		glm::detail::tvec2<T> range( glm::detail::tvec2<T> min, glm::detail::tvec2<T> max )
		{
			return glm::detail::tvec2<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() )
				);
		}

		template < typename T >
		glm::detail::tvec3<T> range( glm::detail::tvec3<T> min, glm::detail::tvec3<T> max )
		{
			return glm::detail::tvec3<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() ),
				range_impl( min.z, max.z, std::is_floating_point<T>() )
				);
		}

		template < typename T >
		glm::detail::tvec4<T> range( glm::detail::tvec4<T> min, glm::detail::tvec4<T> max )
		{
			return glm::detail::tvec4<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() ),
				range_impl( min.z, max.z, std::is_floating_point<T>() ), 
				range_impl( min.w, max.w, std::is_floating_point<T>() ) 
				);
		}

	private:
		static seed_type randomized_seed();

		template <typename T>
		T range_impl( T min, T max, const std::true_type& )
		{
			std::uniform_real_distribution<T> dist( min, max );
			return dist( rng );
		}

		template <typename T>
		T range_impl( T min, T max, const std::false_type& )
		{
			std::uniform_int_distribution<T> dist( min, max );
			return dist( rng );
		}
	private:
		std::mt19937 rng;
	};

}

#endif // NV_RANDOM_HH
