// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LUA_HH
#define NV_LUA_HH

#include <istream>
#include <map>
#include <nv/common.hh>
#include <nv/flags.hh>
#include <string>

struct lua_State;

namespace nv
{
	namespace lua
	{
		const int ref_none  = -2;
		const int ref_nil   = -1;
		const int ret_multi = -1;

		class state;
		typedef int reference;

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class table_guard
		{
		public:
			table_guard( state* lstate, const std::string& table, bool global = true );
			table_guard( state* lstate, const std::string& table, int index, bool global = true );
			table_guard( state* lstate, const std::string& table, const std::string& index, bool global = true );
			table_guard( const table_guard& parent, const std::string& index );
			table_guard( const table_guard& parent, int index );
			bool has_field( const std::string& element );
			std::string get_string( const std::string& element, const std::string& defval = "" );
			char get_char( const std::string& element, char defval = ' ' );
			int get_integer( const std::string& element, int defval = 0 );
			double get_double( const std::string& element, double defval = 0.0 );
			bool get_boolean( const std::string& element, bool defval = false );

			template< uint32 SIZE, typename T >
			flags< SIZE, T > get_flags( const std::string& element )
			{
				flags< SIZE, T > result;
				get_raw_flags( element, result.data(), result.size() );
				return result;
			}

			template< uint32 SIZE, typename T >
			void load_flags( const std::string& element, flags< SIZE, T >& flags )
			{
				get_raw_flags( element, flags.data(), flags.size() );
			}
		private:
			void get_raw_flags( const std::string& element, uint8* data, uint32 count );

			state* L;
			stack_guard m_guard;
		};

		class state
		{
			friend class stack_guard;
			friend class table_guard;
		public:
			explicit state( bool load_libs = false );
			explicit state( lua_State* state );
			bool do_string( const std::string& code, const std::string& name, int rvalues = 0 );
			bool do_stream( std::istream& stream, const std::string& name );
			bool do_file( const std::string& filename );
			int get_stack_size();
			void log_stack();
			lua_State* get_raw();
			reference register_object( object * o );
			void unregister_object( object * o );
			operator lua_State*() { return L; }
			~state();
		private:
			int load_string( const std::string& code, const std::string& name );
			int load_stream( std::istream& stream, const std::string& name );
			int load_file( const std::string& filename );
			int do_current( const std::string& name, int rvalues = 0 );
			bool push( const std::string& path, bool global = true );
			void deep_pointer_copy( int index, void* obj );
		private:
			bool m_owner;
			lua_State* L;
		};

	} // namespace lua

} // namespace nv

#endif // NV_LUA_HH