#include <nv/interface/vertex_buffer.hh>
#include <nv/gl/gl_device.hh>
#include <nv/gfx/image.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/interface/program.hh>
#include <nv/interface/texture2d.hh>
#include <nv/interface/mesh_loader.hh>
#include <nv/io/c_file_system.hh>
#include <nv/formats/obj_loader.hh>
#include <nv/logging.hh>
#include <nv/logger.hh>
#include <nv/math.hh>
#include <nv/string.hh>
#include <nv/types.hh>
#include <nv/interface/mesh.hh>

class application
{
public:
	application();
	bool initialize();
	bool run();
	~application();
protected:
	nv::device*       m_device;
	nv::window*       m_window;
	nv::texture2d*    m_diffuse;
	nv::texture2d*    m_specular;
	nv::clear_state   m_clear_state;
	nv::render_state  m_render_state;

	nv::vertex_array* m_va;
	nv::program*      m_program;
	nv::mesh*         m_mesh;
	nv::uint32        m_count;
};

application::application()
{
	m_device = new nv::gl_device();
	m_window = m_device->create_window( 800, 600 );
	nv::sampler sampler( nv::sampler::LINEAR, nv::sampler::REPEAT );
	
	nv::image_data* sprites = m_device->create_image_data( "diffuse.png" );
	m_diffuse  = m_device->create_texture2d( sprites->get_size(), nv::RGBA, nv::UBYTE, sampler, (void*)sprites->get_data() );
	delete sprites;

	sprites    = m_device->create_image_data( "specular.png" );
	m_specular = m_device->create_texture2d( sprites->get_size(), nv::RGBA, nv::UBYTE, sampler, (void*)sprites->get_data() );
	delete sprites;

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_render_state.depth_test.enabled = true;
	m_render_state.culling.enabled    = true;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;
}

bool application::initialize()
{
	nv::c_file_system fs;
	nv::stream* mesh_file = fs.open( "mesh.obj" );
	nv::mesh_loader* loader = new nv::obj_loader();
	loader->load( *mesh_file );
	m_mesh = loader->release_mesh();
	m_count = loader->get_size();
	delete mesh_file;
	delete loader;

	m_program = m_device->create_program( nv::slurp( "obj.vert" ), nv::slurp( "obj.frag" ) );
	m_va      = m_device->create_vertex_array( m_mesh, &m_program->get_attributes(), nv::STATIC_DRAW );
	return true;
}

bool application::run()
{
	int keypress = 0;

	glm::vec3 move( 0, 0, 0 );

	while(!keypress) 
	{
		m_window->get_context()->clear( m_clear_state );

		glm::mat4 model = glm::translate(glm::mat4(1.0f), glm::vec3(0.0f, 0.0f, 0.0f));
		glm::mat4 view  = glm::lookAt(glm::vec3(6.0f, 8.0f, 6.0f) + move, glm::vec3(0.0f, 4.0f, 0.0f) + move, glm::vec3(0.0, 1.0, 0.0));
		glm::mat4 projection = glm::perspective(60.0f, 1.0f*800.0f/600.0f, 0.1f, 1000.0f);
		glm::mat4 mv         = view * model;
		glm::mat3 normal_matrix = glm::transpose(glm::inverse(glm::mat3(mv)));

		m_diffuse->bind( 0 );
		m_specular->bind( 1 );
		m_program->set_opt_uniform( "nv_m_modelview", mv );
		m_program->set_opt_uniform( "nv_m_projection", projection );
		m_program->set_opt_uniform( "nv_m_normal", normal_matrix );
		m_program->set_uniform( "matrix_mvp", projection * mv );
		m_program->set_uniform( "light_position", glm::vec3(12.0, 12.0, 0) );
		m_program->set_uniform( "light_diffuse",  glm::vec4(0.7,0.7,0.7,1.0) );
		m_program->set_uniform( "light_specular", glm::vec4(1.0,1.0,1.0,1.0) );
		m_program->set_uniform( "custom_color_1", glm::vec3(1.0,0.0,0.0) );
		m_program->set_uniform( "custom_color_2", glm::vec3(0.0,0.0,1.0) );
		m_program->set_uniform( "diffuse", 0 );
		m_program->set_uniform( "specular", 1 );
		m_window->get_context()->draw( nv::TRIANGLES, m_render_state, m_program, m_va, m_count * 3 );
		m_window->swap_buffers();

		nv::io_event event;
		while(m_window->poll_event(event)) 
		{      
			switch (event.type) 
			{
			case nv::EV_QUIT:
				keypress = 1;
				break;
			case nv::EV_KEY:
				if (event.key.pressed)
				{
					switch (event.key.code) 
					{
					case nv::KEY_ESCAPE : keypress = 1; break;
					case nv::KEY_LEFT   : move.x = move.x - 1.0f; break;
					case nv::KEY_RIGHT  : move.x = move.x + 1.0f; break;
					case nv::KEY_UP     : move.z = move.z - 1.0f; break;
					case nv::KEY_DOWN   : move.z = move.z + 1.0f; break;
					default: break;
					}
				}
				break;
			default: break;
			}
		}
	}
	return true;
}

application::~application()
{
	delete m_program;
	delete m_mesh;
	delete m_diffuse;
	delete m_specular;
	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

