// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file array2d.hh
 * @author Kornel Kisielewicz
 * @brief 2D array
 */

// TODO: implement copy on resize! See image::set_region
// TODO: implement set sub_array or something like that
// TODO: make it work with the stl allocator

#ifndef NV_ARRAY2D_HH
#define NV_ARRAY2D_HH

#include <nv/common.hh>
#include <nv/math.hh>

namespace nv
{
	typedef ivec2 coord2d;

	template < typename T >
	class array2d
	{
	public:
		typedef T                 value_type;
		typedef value_type&       reference;
		typedef const value_type& const_reference;
		typedef value_type*       pointer;
		typedef const value_type* const_pointer;
		typedef pointer           iterator;
		typedef const_pointer     const_iterator;

		array2d() : m_size(), m_data( nullptr ) {}
		array2d( const ivec2& asize ) : m_size(), m_data( nullptr ) { resize( asize ); }
		
		const ivec2& size() const { return m_size; }
		pointer data() { return m_data; }
		const_pointer data() const { return m_data; }

		void resize( const ivec2& new_size ) 
		{
			pointer new_data = new value_type[ new_size.x * new_size.y ];
			if ( m_data != nullptr )
			{
				// TODO: implement copy! See image::set_region
				delete[] m_data;
			}
			m_data = new_data;
			m_size = new_size;
		}

		iterator       begin()       { return m_data; }
		const_iterator begin() const { return m_data; }
		iterator       end()         { return m_data + ( m_size.x * m_size.y ); }
		const_iterator end() const   { return m_data + ( m_size.x * m_size.y ); }

		inline const_reference operator() ( sint32 x, sint32 y ) const
		{
			NV_ASSERT( x >= 0 && y >= 0 && x < m_size.x && y < m_size.y, "Bad parameters passed to array2d()" );
			return m_data[ y * m_size.x + x ];
		}
		inline reference operator() ( sint32 x, sint32 y )
		{
			NV_ASSERT( x >= 0 && y >= 0 && x < m_size.x && y < m_size.y, "Bad parameters passed to array2d()" );
			return m_data[ y * m_size.x + x ];
		}
		inline const_reference operator[] ( const ivec2& c ) const 
		{
			NV_ASSERT( c.x >= 0 && c.y >= 0 && x < m_size.x && y < m_size.y, "Bad parameters passed to array2d[]" );
			return m_data[ c.y * m_size.x + c.x ];
		}
		inline reference operator[] ( const ivec2& c )
		{
			NV_ASSERT( x >= 0 && y >= 0 && x < m_size.x && y < m_size.y, "Bad parameters passed to array2d[]" );
			return m_data[ c.y * m_size.x + c.x ];
		}

		~array2d()
		{
			if ( m_data != nullptr )
			{
				delete[] m_data;
			}
		}
	protected:
		pointer m_data;
		ivec2   m_size;
	};

}

#endif // NV_ARRAY2D_HH
