// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

// TODO : Force no resize on the buffer_slice!
// TODO : Multiple update support?

#ifndef NV_CACHED_BUFFER_HH
#define NV_CACHED_BUFFER_HH

#include <nv/common.hh>
#include <nv/interface/vertex_buffer.hh>

namespace nv
{

	template< typename T > class cached_buffer;

	template< typename T >
	class buffer_slice
	{
	public:
		typedef cached_buffer<T> cache;
		typedef std::vector<T>  vector;

		buffer_slice( cache* c ) 
			: m_cache( c ), m_offset( 0 ), m_cached_size( 0 ), m_locked( true )
		{
			m_cache->reset();
		}
		bool commit( bool force_changed = false )
		{
			bool resized = force_changed || ( m_cached_size != m_data.size() );
			bool result  = m_cache->commit( m_data, m_locked, resized, m_offset );
			m_cached_size = m_data.size();
			m_locked      = false;
			return result;
		}
		
		template < typename V >
		bool commit_offset( bool force_changed, V value )
		{
			bool locked = m_locked;
			bool result = commit( force_changed );
			if ( locked || result )
			{
				// funny, but this is what GCC expects
				m_cache->template add_offset<T>( m_data.size(), m_offset, static_cast<T>(value) );
			}
			return result;
		}

		bool is_locked() const
		{
			return m_locked;
		}

		size_t get_offset() const
		{
			return m_offset;
		}

		vector& lock()
		{
			m_locked = true;
			return m_data;
		}
		const vector& data() 
		{ 
			return m_data; 
		}
		~buffer_slice()
		{
			m_cache->reset();
		}
	public:
		vector m_data;
		cache* m_cache;
		size_t m_offset;
		size_t m_cached_size;
		bool   m_locked;
	};

	template< typename T >
	class cached_buffer
	{
	public:
		typedef std::vector<T>  vector;
		typedef T               value_type;
		static const size_t value_type_size = sizeof(T);

		cached_buffer( device* dev, buffer_hint hint, size_t initial_size, bool is_vertex = true ) 
			: m_device( dev )
			, m_buffer( nullptr )
			, m_hint( hint )
			, m_full_update( true )
			, m_is_vertex( is_vertex )
			, m_data()
			, m_min(0)
			, m_max(0)
		{ 
			create_buffer( initial_size );
		}

		bool commit( const vector& bv, bool updated, bool resized, size_t& offset )
		{
			if ( !m_full_update && resized )
			{
				m_data.erase( m_data.begin() + offset, m_data.end() );
				m_min = glm::min<int>( m_min, offset );
				m_full_update = true;
			}
			if ( m_full_update )
			{
				offset = m_data.size();
				m_data.insert( m_data.end(), bv.cbegin(), bv.cend() );
			}
			else if ( updated )
			{
				std::copy( bv.cbegin(), bv.cend(), m_data.begin() + offset );
				m_min = glm::min<size_t>( m_min, offset );
				m_max = glm::max<size_t>( m_max, offset + bv.size() );
			}
			return m_full_update;
		}

		template < typename V > 
		void add_offset( size_t size, size_t offset, V value )
		{
			if ( size == 0 ) return;
			T* ptr  = m_data.data() + offset;
			T* pend = ptr + size;
			for ( ; ptr != pend; ptr++ )
			{
				*ptr += value;
			}
		}

		void reset()
		{
			m_data.clear();
			m_full_update = true;
			m_min = 0;
			m_max = 0;
		}

		/**
		 * Returns true if buffer has been recreated
		 */
		bool commit()
		{
			bool result = false;
			size_t bsize = (size_t)get_max_size();
			if ( m_data.size() > bsize )
			{
				while( m_data.size() > bsize ) bsize *= 2;
				create_buffer( bsize );
				m_full_update = true;
				m_min  = 0;
				result = true;
			}
			if ( m_full_update ) m_max = m_data.size();
			if ( m_max > 0 )
			{
				m_buffer->bind();
				int offset = m_min * value_type_size;
				int size   = (m_max-m_min) * value_type_size;
				m_buffer->update( m_data.data() + m_min, offset, size );
				m_buffer->unbind();
			}
			m_full_update = false;
			m_min = get_max_size();
			m_max = 0;
			return result;
		}

		int get_max_size() const
		{ 
			return m_buffer->get_size() / value_type_size;
		}

		int get_size() const
		{ 
			return m_data.size(); 
		}

		buffer* get_buffer()
		{ 
			return m_buffer; 
		}

		virtual ~cached_buffer() 
		{
			delete m_buffer;
		}
	private:
		void create_buffer( int size )
		{
			delete m_buffer;
			if ( m_is_vertex )
				m_buffer = m_device->create_vertex_buffer( m_hint, size * value_type_size, nullptr );
			else
				m_buffer = m_device->create_index_buffer( m_hint, size * value_type_size, nullptr );
		}
	private:
		device*     m_device;
		buffer*     m_buffer;
		buffer_hint m_hint;
		bool        m_full_update;
		bool        m_is_vertex;
		vector      m_data;

		size_t      m_min;
		size_t      m_max;
	};

	template< typename T, typename I = uint16 > class indexed_cached_buffer;

	template< typename T, typename I = uint16 >
	class indexed_buffer_slice
	{
	public:
		typedef indexed_cached_buffer<T,I> cache;
		typedef buffer_slice<T> vertex_slice;
		typedef buffer_slice<I> index_slice;
		typedef typename vertex_slice::vector vertex_vector;
		typedef typename index_slice::vector  index_vector;

		indexed_buffer_slice( cache* c )
			: m_vertex_slice( &c->get_vertex_cache() )
			, m_index_slice( &c->get_index_cache() )
			, m_cache( c )
		{
		}

		void commit()
		{
			bool changed = m_vertex_slice.commit();
			m_index_slice.commit_offset( changed, m_vertex_slice.get_offset() );
		}
		vertex_vector& lock_vertices()
		{
			return m_vertex_slice.lock();
		}
		index_vector& lock_indices()
		{
			return m_index_slice.lock();
		}
		const vertex_vector& vertex_data() 
		{ 
			return m_vertex_slice.data();
		}
		const index_vector& index_data() 
		{ 
			return m_index_slice.data();
		}

	private:
		vertex_slice m_vertex_slice;
		index_slice  m_index_slice;
		cache*       m_cache;
	};

	template< typename T, typename I >
	class indexed_cached_buffer
	{
	public:
		typedef cached_buffer<T> cached_vertex_buffer;
		typedef cached_buffer<I> cached_index_buffer;
		static const int value_type_size = sizeof(T);

		indexed_cached_buffer( device* dev, buffer_hint hint, int initial_size, int initial_index_size )
			: m_vertex_buffer( dev, hint, initial_size, true )
			, m_index_buffer( dev, hint, initial_index_size, false )
		{

		}

		void reset()
		{
			m_vertex_buffer.reset();
			m_index_buffer.reset();
		}

		bool commit()
		{
			bool vresult = m_vertex_buffer.commit();
			bool iresult = m_index_buffer.commit();
			return vresult || iresult;
		}

		int get_max_vertex_size() const { return m_vertex_buffer.get_max_size(); }
		int get_max_index_size()  const { return m_index_buffer.get_max_size(); }
		int get_vertex_size()     const { return m_vertex_buffer.get_size(); }
		int get_index_size()      const { return m_index_buffer.get_size(); }
		vertex_buffer* get_vertex_buffer() { return static_cast<vertex_buffer*>( m_vertex_buffer.get_buffer() ); }
		index_buffer*  get_index_buffer()  { return static_cast<index_buffer*>( m_index_buffer.get_buffer() ); }
		cached_vertex_buffer& get_vertex_cache() { return m_vertex_buffer; }
		cached_index_buffer&  get_index_cache()  { return m_index_buffer; }

	private:
		cached_vertex_buffer m_vertex_buffer;
		cached_index_buffer  m_index_buffer;
	};


} // namespace nv

#endif // NV_CACHED_BUFFER_HH
