// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
 * @file vertex_buffer.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Vertex buffer class
 */

#ifndef NV_VERTEX_BUFFER_HH
#define NV_VERTEX_BUFFER_HH

#include <nv/common.hh>
#include <nv/types.hh>
#include <unordered_map>
#include <string>

namespace nv
{

	enum buffer_hint
	{
		STATIC_DRAW,
		STREAM_DRAW,
		DYNAMIC_DRAW
	};

	class buffer
	{
	public:
		buffer( buffer_hint hint, int size ) { m_size = size; m_hint = hint; }
		virtual void update( void* data, int offset, int size ) = 0;
		virtual void bind() = 0;
		virtual void unbind() = 0;
		virtual bool is_valid() const = 0;
		int get_size() const { return m_size; };
		buffer_hint get_hint() const { return m_hint; };
	protected:
		int         m_size;
		buffer_hint m_hint;
	};

	class vertex_buffer : public buffer
	{
	public:
		vertex_buffer( buffer_hint hint, int size ) : buffer( hint, size ) {}
	};

	class index_buffer  : public buffer 
	{
	public:
		index_buffer( buffer_hint hint, int size ) : buffer( hint, size ) {}
	};

	class vertex_buffer_attribute
	{
	public:
		vertex_buffer_attribute( vertex_buffer* buffer, datatype datatype, int components, int offset = 0, int stride = 0, bool owner = true )
			: m_buffer( buffer ), m_datatype( datatype ), m_components( components ), m_offset( offset ), m_stride( stride ), m_owner( owner ) {}

		vertex_buffer* get_buffer() const { return m_buffer; }
		datatype get_datatype() const { return m_datatype; }
		int get_components() const { return m_components; }
		int get_offset() const { return m_offset; }
		int get_stride() const { return m_stride; }

		~vertex_buffer_attribute()
		{
			if (m_owner)
			{
				delete m_buffer;
			}
		}
	protected:
		vertex_buffer* m_buffer;
		datatype       m_datatype;
		int  m_components;
		int  m_offset;
		int  m_stride;
		bool m_owner;
	};

	typedef std::unordered_map< int, vertex_buffer_attribute* > vertex_buffer_attribute_map;

	class vertex_array
	{
	public:
		vertex_array() : m_map(), m_index( nullptr ) {}
		void add_vertex_buffer( int location, vertex_buffer* buffer, datatype datatype, int components, int offset = 0, int stride = 0, bool owner = true ) 
		{
			m_map[ location ] = new vertex_buffer_attribute( buffer, datatype, components, offset, stride, owner );
		};
		void set_index_buffer( index_buffer* buffer ) { m_index = buffer; }
		virtual void bind() = 0;
		virtual void unbind() = 0;
		~vertex_array() { 
			for ( vertex_buffer_attribute_map::iterator i = m_map.begin(); 	i != m_map.end(); ++i ) 
			{
				delete i->second;
			}
			if (m_index) delete m_index; 
		}
	protected:
		vertex_buffer_attribute_map m_map;
		index_buffer* m_index;
	};


} // namespace nv

#endif // NV_VERTEX_BUFFER_HH
