// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_LIB_SDL_HH
#define NV_LIB_SDL_HH

#include <nv/common.hh>

#define NV_SDL_DYNAMIC
//#define NV_SDL_SHARED

#if NV_PLATFORM == NV_WINDOWS
#	define NV_SDL_PATH "SDL.dll"
#elif NV_PLATFORM == NV_APPLE
#	define NV_SDL_PATH "SDL.framework/SDL"
#else
#	define NV_SDL_PATH "libSDL-1.2.so.0"
#endif

#include <stdint.h>
#include <stdio.h>

extern "C" {

#if NV_PLATFORM == NV_WINDOWS 
#	define NV_SDL_APIENTRY __cdecl
#else
#	define NV_SDL_APIENTRY 
#endif

#if defined(NV_SDL_SHARED) && (NV_PLATFORM == NV_WINDOWS)
#	define NV_SDL_API __declspec(dllimport)
#else
#	define NV_SDL_API extern
#endif

#if defined(NV_SDL_DYNAMIC)
#	define NV_SDL_FUN( rtype, fname, fparams ) NV_SDL_API rtype (NV_SDL_APIENTRY *fname) fparams
#else
#	define NV_SDL_FUN( rtype, fname, fparams ) NV_SDL_API rtype NV_SDL_APIENTRY fname fparams
#endif

/** REMOVE THIS */
#define DECLSPEC NV_SDL_API
#define SDLCALL

/* SDL_stdinc.h types */
typedef enum {
	SDL_FALSE = 0,
	SDL_TRUE  = 1
} SDL_bool;

typedef int8_t		Sint8;
typedef uint8_t		Uint8;
typedef int16_t		Sint16;
typedef uint16_t	Uint16;
typedef int32_t		Sint32;
typedef uint32_t	Uint32;
typedef int64_t		Sint64;
typedef uint64_t	Uint64;

static_assert(sizeof(Uint8) == 1, "Uint8 has wrong size!" );
static_assert(sizeof(Sint8) == 1, "Sint8 has wrong size!");
static_assert(sizeof(Uint16) == 2, "Uint16 has wrong size!");
static_assert(sizeof(Sint16) == 2, "Sint16 has wrong size!");
static_assert(sizeof(Uint32) == 4, "Uint32 has wrong size!");
static_assert(sizeof(Sint32) == 4, "Sint32 has wrong size!");
static_assert(sizeof(Uint64) == 8, "Uint64 has wrong size!");
static_assert(sizeof(Sint64) == 8, "Sint64 has wrong size!");

#define RW_SEEK_SET	0
#define RW_SEEK_CUR	1
#define RW_SEEK_END	2

/* SDL.h defines */

#define	SDL_INIT_TIMER		0x00000001
#define SDL_INIT_AUDIO		0x00000010
#define SDL_INIT_VIDEO		0x00000020
#define SDL_INIT_CDROM		0x00000100
#define SDL_INIT_JOYSTICK	0x00000200
#define SDL_INIT_NOPARACHUTE	0x00100000
#define SDL_INIT_EVENTTHREAD	0x01000000
#define SDL_INIT_EVERYTHING	0x0000FFFF 

/* SDL_error.h defines */
typedef enum {
	SDL_ENOMEM,
	SDL_EFREAD,
	SDL_EFWRITE,
	SDL_EFSEEK,
	SDL_UNSUPPORTED,
	SDL_LASTERROR
} SDL_errorcode;

/* SDL_rwops.h defines */
typedef struct SDL_RWops {
	int (SDLCALL *seek)(struct SDL_RWops *context, int offset, int whence);
	int (SDLCALL *read)(struct SDL_RWops *context, void *ptr, int size, int maxnum);
	int (SDLCALL *write)(struct SDL_RWops *context, const void *ptr, int size, int num);
	int (SDLCALL *close)(struct SDL_RWops *context);

	Uint32 type;
	union {
#if NV_PLATFORM == NV_WINDOWS 
	    struct {
		int   append;
		void *h;
		struct {
		    void *data;
		    int size;
		    int left;
		} buffer;
	    } win32io;
#endif
	    struct {
		int autoclose;
	 	FILE *fp;
	    } stdio;
	    struct {
		Uint8 *base;
	 	Uint8 *here;
		Uint8 *stop;
	    } mem;
	    struct {
		void *data1;
	    } unknown;
	} hidden;

} SDL_RWops;

/* SDL_video.h defines */
#define SDL_ALPHA_OPAQUE 255
#define SDL_ALPHA_TRANSPARENT 0

typedef struct SDL_Rect {
	Sint16 x, y;
	Uint16 w, h;
} SDL_Rect;

typedef struct SDL_Color {
	Uint8 r;
	Uint8 g;
	Uint8 b;
	Uint8 unused;
} SDL_Color;
#define SDL_Colour SDL_Color

typedef struct SDL_Palette {
	int       ncolors;
	SDL_Color *colors;
} SDL_Palette;

typedef struct SDL_PixelFormat {
	SDL_Palette *palette;
	Uint8  BitsPerPixel;
	Uint8  BytesPerPixel;
	Uint8  Rloss;
	Uint8  Gloss;
	Uint8  Bloss;
	Uint8  Aloss;
	Uint8  Rshift;
	Uint8  Gshift;
	Uint8  Bshift;
	Uint8  Ashift;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;

	Uint32 colorkey;
	Uint8  alpha;
} SDL_PixelFormat;

typedef struct SDL_Surface {
	Uint32 flags;
	SDL_PixelFormat *format;
	int w, h;
	Uint16 pitch;
	void *pixels;
	int offset;	
	struct private_hwdata *hwdata;
	SDL_Rect clip_rect;
	Uint32 unused1;
	Uint32 locked;
	struct SDL_BlitMap *map;
	unsigned int format_version;
	int refcount;
} SDL_Surface;

#define SDL_SWSURFACE	0x00000000	
#define SDL_HWSURFACE	0x00000001	
#define SDL_ASYNCBLIT	0x00000004	
#define SDL_ANYFORMAT	0x10000000	
#define SDL_HWPALETTE	0x20000000	
#define SDL_DOUBLEBUF	0x40000000	
#define SDL_FULLSCREEN	0x80000000	
#define SDL_OPENGL      0x00000002  
#define SDL_OPENGLBLIT	0x0000000A	
#define SDL_RESIZABLE	0x00000010	
#define SDL_NOFRAME	0x00000020	
#define SDL_HWACCEL	0x00000100	
#define SDL_SRCCOLORKEY	0x00001000	
#define SDL_RLEACCELOK	0x00002000	
#define SDL_RLEACCEL	0x00004000	
#define SDL_SRCALPHA	0x00010000	
#define SDL_PREALLOC	0x01000000	

#define SDL_MUSTLOCK(surface)	\
  (surface->offset ||		\
  ((surface->flags & (SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_RLEACCEL)) != 0))

typedef int (*SDL_blit)(struct SDL_Surface *src, SDL_Rect *srcrect, struct SDL_Surface *dst, SDL_Rect *dstrect);

typedef struct SDL_VideoInfo {
	Uint32 hw_available :1;
	Uint32 wm_available :1;
	Uint32 UnusedBits1  :6;
	Uint32 UnusedBits2  :1;
	Uint32 blit_hw      :1;
	Uint32 blit_hw_CC   :1;
	Uint32 blit_hw_A    :1;
	Uint32 blit_sw      :1;
	Uint32 blit_sw_CC   :1;
	Uint32 blit_sw_A    :1;
	Uint32 blit_fill    :1;
	Uint32 UnusedBits3  :16;
	Uint32 video_mem;
	SDL_PixelFormat *vfmt;
	int    current_w;
	int    current_h;
} SDL_VideoInfo;

#define SDL_YV12_OVERLAY  0x32315659
#define SDL_IYUV_OVERLAY  0x56555949
#define SDL_YUY2_OVERLAY  0x32595559
#define SDL_UYVY_OVERLAY  0x59565955
#define SDL_YVYU_OVERLAY  0x55595659

typedef struct SDL_Overlay {
	Uint32 format;
	int w, h;
	int planes;
	Uint16 *pitches;
	Uint8 **pixels;
	struct private_yuvhwfuncs *hwfuncs;
	struct private_yuvhwdata *hwdata;
	Uint32 hw_overlay :1;
	Uint32 UnusedBits :31;
} SDL_Overlay;

typedef enum {
    SDL_GL_RED_SIZE,
    SDL_GL_GREEN_SIZE,
    SDL_GL_BLUE_SIZE,
    SDL_GL_ALPHA_SIZE,
    SDL_GL_BUFFER_SIZE,
    SDL_GL_DOUBLEBUFFER,
    SDL_GL_DEPTH_SIZE,
    SDL_GL_STENCIL_SIZE,
    SDL_GL_ACCUM_RED_SIZE,
    SDL_GL_ACCUM_GREEN_SIZE,
    SDL_GL_ACCUM_BLUE_SIZE,
    SDL_GL_ACCUM_ALPHA_SIZE,
    SDL_GL_STEREO,
    SDL_GL_MULTISAMPLEBUFFERS,
    SDL_GL_MULTISAMPLESAMPLES,
    SDL_GL_ACCELERATED_VISUAL,
    SDL_GL_SWAP_CONTROL
} SDL_GLattr;

#define SDL_LOGPAL 0x01
#define SDL_PHYSPAL 0x02

typedef enum {
	SDL_GRAB_QUERY = -1,
	SDL_GRAB_OFF = 0,
	SDL_GRAB_ON = 1,
	SDL_GRAB_FULLSCREEN	/**< Used internally */
} SDL_GrabMode;

/* SDL_audio.h defines */
typedef struct SDL_AudioSpec {
	int freq;
	Uint16 format;
	Uint8  channels;
	Uint8  silence;
	Uint16 samples;
	Uint16 padding;
	Uint32 size;
	void (SDLCALL *callback)(void *userdata, Uint8 *stream, int len);
	void  *userdata;
} SDL_AudioSpec;

#define AUDIO_U8	0x0008
#define AUDIO_S8	0x8008
#define AUDIO_U16LSB	0x0010	
#define AUDIO_S16LSB	0x8010	
#define AUDIO_U16MSB	0x1010	
#define AUDIO_S16MSB	0x9010
#define AUDIO_U16	AUDIO_U16LSB
#define AUDIO_S16	AUDIO_S16LSB

/* NOTE : assuming LSB! */
#define AUDIO_U16SYS	AUDIO_U16LSB
#define AUDIO_S16SYS	AUDIO_S16LSB

typedef struct SDL_AudioCVT {
	int needed;
	Uint16 src_format;
	Uint16 dst_format;
	double rate_incr;
	Uint8 *buf;		
	int    len;		
	int    len_cvt;	
	int    len_mult;
	double len_ratio;
	void (SDLCALL *filters[10])(struct SDL_AudioCVT *cvt, Uint16 format);
	int filter_index;
} SDL_AudioCVT; 
	
typedef enum {
	SDL_AUDIO_STOPPED = 0,
	SDL_AUDIO_PLAYING,
	SDL_AUDIO_PAUSED
} SDL_audiostatus;

#define SDL_MIX_MAXVOLUME 128

/* SDL_active.h defines */
#define SDL_APPMOUSEFOCUS	0x01
#define SDL_APPINPUTFOCUS	0x02
#define SDL_APPACTIVE		0x04

/* SDL_keyboard.h defines */
typedef enum {
	SDLK_UNKNOWN		= 0,
	SDLK_FIRST		= 0,
	SDLK_BACKSPACE		= 8,
	SDLK_TAB		= 9,
	SDLK_CLEAR		= 12,
	SDLK_RETURN		= 13,
	SDLK_PAUSE		= 19,
	SDLK_ESCAPE		= 27,
	SDLK_SPACE		= 32,
	SDLK_EXCLAIM		= 33,
	SDLK_QUOTEDBL		= 34,
	SDLK_HASH		= 35,
	SDLK_DOLLAR		= 36,
	SDLK_AMPERSAND		= 38,
	SDLK_QUOTE		= 39,
	SDLK_LEFTPAREN		= 40,
	SDLK_RIGHTPAREN		= 41,
	SDLK_ASTERISK		= 42,
	SDLK_PLUS		= 43,
	SDLK_COMMA		= 44,
	SDLK_MINUS		= 45,
	SDLK_PERIOD		= 46,
	SDLK_SLASH		= 47,
	SDLK_0			= 48,
	SDLK_1			= 49,
	SDLK_2			= 50,
	SDLK_3			= 51,
	SDLK_4			= 52,
	SDLK_5			= 53,
	SDLK_6			= 54,
	SDLK_7			= 55,
	SDLK_8			= 56,
	SDLK_9			= 57,
	SDLK_COLON		= 58,
	SDLK_SEMICOLON		= 59,
	SDLK_LESS		= 60,
	SDLK_EQUALS		= 61,
	SDLK_GREATER		= 62,
	SDLK_QUESTION		= 63,
	SDLK_AT			= 64,
	SDLK_LEFTBRACKET	= 91,
	SDLK_BACKSLASH		= 92,
	SDLK_RIGHTBRACKET	= 93,
	SDLK_CARET		= 94,
	SDLK_UNDERSCORE		= 95,
	SDLK_BACKQUOTE		= 96,
	SDLK_a			= 97,
	SDLK_b			= 98,
	SDLK_c			= 99,
	SDLK_d			= 100,
	SDLK_e			= 101,
	SDLK_f			= 102,
	SDLK_g			= 103,
	SDLK_h			= 104,
	SDLK_i			= 105,
	SDLK_j			= 106,
	SDLK_k			= 107,
	SDLK_l			= 108,
	SDLK_m			= 109,
	SDLK_n			= 110,
	SDLK_o			= 111,
	SDLK_p			= 112,
	SDLK_q			= 113,
	SDLK_r			= 114,
	SDLK_s			= 115,
	SDLK_t			= 116,
	SDLK_u			= 117,
	SDLK_v			= 118,
	SDLK_w			= 119,
	SDLK_x			= 120,
	SDLK_y			= 121,
	SDLK_z			= 122,
	SDLK_DELETE		= 127,
	SDLK_WORLD_0		= 160,
	SDLK_WORLD_1		= 161,
	SDLK_WORLD_2		= 162,
	SDLK_WORLD_3		= 163,
	SDLK_WORLD_4		= 164,
	SDLK_WORLD_5		= 165,
	SDLK_WORLD_6		= 166,
	SDLK_WORLD_7		= 167,
	SDLK_WORLD_8		= 168,
	SDLK_WORLD_9		= 169,
	SDLK_WORLD_10		= 170,
	SDLK_WORLD_11		= 171,
	SDLK_WORLD_12		= 172,
	SDLK_WORLD_13		= 173,
	SDLK_WORLD_14		= 174,
	SDLK_WORLD_15		= 175,
	SDLK_WORLD_16		= 176,
	SDLK_WORLD_17		= 177,
	SDLK_WORLD_18		= 178,
	SDLK_WORLD_19		= 179,
	SDLK_WORLD_20		= 180,
	SDLK_WORLD_21		= 181,
	SDLK_WORLD_22		= 182,
	SDLK_WORLD_23		= 183,
	SDLK_WORLD_24		= 184,
	SDLK_WORLD_25		= 185,
	SDLK_WORLD_26		= 186,
	SDLK_WORLD_27		= 187,
	SDLK_WORLD_28		= 188,
	SDLK_WORLD_29		= 189,
	SDLK_WORLD_30		= 190,
	SDLK_WORLD_31		= 191,
	SDLK_WORLD_32		= 192,
	SDLK_WORLD_33		= 193,
	SDLK_WORLD_34		= 194,
	SDLK_WORLD_35		= 195,
	SDLK_WORLD_36		= 196,
	SDLK_WORLD_37		= 197,
	SDLK_WORLD_38		= 198,
	SDLK_WORLD_39		= 199,
	SDLK_WORLD_40		= 200,
	SDLK_WORLD_41		= 201,
	SDLK_WORLD_42		= 202,
	SDLK_WORLD_43		= 203,
	SDLK_WORLD_44		= 204,
	SDLK_WORLD_45		= 205,
	SDLK_WORLD_46		= 206,
	SDLK_WORLD_47		= 207,
	SDLK_WORLD_48		= 208,
	SDLK_WORLD_49		= 209,
	SDLK_WORLD_50		= 210,
	SDLK_WORLD_51		= 211,
	SDLK_WORLD_52		= 212,
	SDLK_WORLD_53		= 213,
	SDLK_WORLD_54		= 214,
	SDLK_WORLD_55		= 215,
	SDLK_WORLD_56		= 216,
	SDLK_WORLD_57		= 217,
	SDLK_WORLD_58		= 218,
	SDLK_WORLD_59		= 219,
	SDLK_WORLD_60		= 220,
	SDLK_WORLD_61		= 221,
	SDLK_WORLD_62		= 222,
	SDLK_WORLD_63		= 223,
	SDLK_WORLD_64		= 224,
	SDLK_WORLD_65		= 225,
	SDLK_WORLD_66		= 226,
	SDLK_WORLD_67		= 227,
	SDLK_WORLD_68		= 228,
	SDLK_WORLD_69		= 229,
	SDLK_WORLD_70		= 230,
	SDLK_WORLD_71		= 231,
	SDLK_WORLD_72		= 232,
	SDLK_WORLD_73		= 233,
	SDLK_WORLD_74		= 234,
	SDLK_WORLD_75		= 235,
	SDLK_WORLD_76		= 236,
	SDLK_WORLD_77		= 237,
	SDLK_WORLD_78		= 238,
	SDLK_WORLD_79		= 239,
	SDLK_WORLD_80		= 240,
	SDLK_WORLD_81		= 241,
	SDLK_WORLD_82		= 242,
	SDLK_WORLD_83		= 243,
	SDLK_WORLD_84		= 244,
	SDLK_WORLD_85		= 245,
	SDLK_WORLD_86		= 246,
	SDLK_WORLD_87		= 247,
	SDLK_WORLD_88		= 248,
	SDLK_WORLD_89		= 249,
	SDLK_WORLD_90		= 250,
	SDLK_WORLD_91		= 251,
	SDLK_WORLD_92		= 252,
	SDLK_WORLD_93		= 253,
	SDLK_WORLD_94		= 254,
	SDLK_WORLD_95		= 255,
	SDLK_KP0		= 256,
	SDLK_KP1		= 257,
	SDLK_KP2		= 258,
	SDLK_KP3		= 259,
	SDLK_KP4		= 260,
	SDLK_KP5		= 261,
	SDLK_KP6		= 262,
	SDLK_KP7		= 263,
	SDLK_KP8		= 264,
	SDLK_KP9		= 265,
	SDLK_KP_PERIOD		= 266,
	SDLK_KP_DIVIDE		= 267,
	SDLK_KP_MULTIPLY	= 268,
	SDLK_KP_MINUS		= 269,
	SDLK_KP_PLUS		= 270,
	SDLK_KP_ENTER		= 271,
	SDLK_KP_EQUALS		= 272,
	SDLK_UP			= 273,
	SDLK_DOWN		= 274,
	SDLK_RIGHT		= 275,
	SDLK_LEFT		= 276,
	SDLK_INSERT		= 277,
	SDLK_HOME		= 278,
	SDLK_END		= 279,
	SDLK_PAGEUP		= 280,
	SDLK_PAGEDOWN		= 281,
	SDLK_F1			= 282,
	SDLK_F2			= 283,
	SDLK_F3			= 284,
	SDLK_F4			= 285,
	SDLK_F5			= 286,
	SDLK_F6			= 287,
	SDLK_F7			= 288,
	SDLK_F8			= 289,
	SDLK_F9			= 290,
	SDLK_F10		= 291,
	SDLK_F11		= 292,
	SDLK_F12		= 293,
	SDLK_F13		= 294,
	SDLK_F14		= 295,
	SDLK_F15		= 296,
	SDLK_NUMLOCK		= 300,
	SDLK_CAPSLOCK		= 301,
	SDLK_SCROLLOCK		= 302,
	SDLK_RSHIFT		= 303,
	SDLK_LSHIFT		= 304,
	SDLK_RCTRL		= 305,
	SDLK_LCTRL		= 306,
	SDLK_RALT		= 307,
	SDLK_LALT		= 308,
	SDLK_RMETA		= 309,
	SDLK_LMETA		= 310,
	SDLK_LSUPER		= 311,
	SDLK_RSUPER		= 312,
	SDLK_MODE		= 313,
	SDLK_COMPOSE		= 314,
	SDLK_HELP		= 315,
	SDLK_PRINT		= 316,
	SDLK_SYSREQ		= 317,
	SDLK_BREAK		= 318,
	SDLK_MENU		= 319,
	SDLK_POWER		= 320,
	SDLK_EURO		= 321,
	SDLK_UNDO		= 322,
	SDLK_LAST
} SDLKey;

typedef enum {
	KMOD_NONE  = 0x0000,
	KMOD_LSHIFT= 0x0001,
	KMOD_RSHIFT= 0x0002,
	KMOD_LCTRL = 0x0040,
	KMOD_RCTRL = 0x0080,
	KMOD_LALT  = 0x0100,
	KMOD_RALT  = 0x0200,
	KMOD_LMETA = 0x0400,
	KMOD_RMETA = 0x0800,
	KMOD_NUM   = 0x1000,
	KMOD_CAPS  = 0x2000,
	KMOD_MODE  = 0x4000,
	KMOD_RESERVED = 0x8000
} SDLMod;

#define KMOD_CTRL	(KMOD_LCTRL|KMOD_RCTRL)
#define KMOD_SHIFT	(KMOD_LSHIFT|KMOD_RSHIFT)
#define KMOD_ALT	(KMOD_LALT|KMOD_RALT)
#define KMOD_META	(KMOD_LMETA|KMOD_RMETA)

#define SDL_DEFAULT_REPEAT_DELAY	500
#define SDL_DEFAULT_REPEAT_INTERVAL	30

typedef struct SDL_keysym {
	Uint8 scancode;
	SDLKey sym;
	SDLMod mod;
	Uint16 unicode;
} SDL_keysym;

#define SDL_ALL_HOTKEYS		0xFFFFFFFF

/* SDL_mouse.h defines */
typedef struct WMcursor WMcursor;
typedef struct SDL_Cursor {
	SDL_Rect area;	
	Sint16 hot_x, hot_y;
	Uint8 *data;
	Uint8 *mask;
	Uint8 *save[2];	
	WMcursor *wm_cursor;
} SDL_Cursor;
 
#define SDL_BUTTON(X)		(1 << ((X)-1))
#define SDL_BUTTON_LEFT		1
#define SDL_BUTTON_MIDDLE	2
#define SDL_BUTTON_RIGHT	3
#define SDL_BUTTON_WHEELUP	4
#define SDL_BUTTON_WHEELDOWN	5
#define SDL_BUTTON_X1		6
#define SDL_BUTTON_X2		7
#define SDL_BUTTON_LMASK	SDL_BUTTON(SDL_BUTTON_LEFT)
#define SDL_BUTTON_MMASK	SDL_BUTTON(SDL_BUTTON_MIDDLE)
#define SDL_BUTTON_RMASK	SDL_BUTTON(SDL_BUTTON_RIGHT)
#define SDL_BUTTON_X1MASK	SDL_BUTTON(SDL_BUTTON_X1)
#define SDL_BUTTON_X2MASK	SDL_BUTTON(SDL_BUTTON_X2) 

/* SDL_joystick.h defines */
struct _SDL_Joystick;
typedef struct _SDL_Joystick SDL_Joystick; 

#define SDL_HAT_CENTERED	0x00
#define SDL_HAT_UP		0x01
#define SDL_HAT_RIGHT		0x02
#define SDL_HAT_DOWN		0x04
#define SDL_HAT_LEFT		0x08
#define SDL_HAT_RIGHTUP		(SDL_HAT_RIGHT|SDL_HAT_UP)
#define SDL_HAT_RIGHTDOWN	(SDL_HAT_RIGHT|SDL_HAT_DOWN)
#define SDL_HAT_LEFTUP		(SDL_HAT_LEFT|SDL_HAT_UP)
#define SDL_HAT_LEFTDOWN	(SDL_HAT_LEFT|SDL_HAT_DOWN)

/* SDL_event.h defines */
#define SDL_RELEASED	0
#define SDL_PRESSED	1

typedef enum {
       SDL_NOEVENT = 0,
       SDL_ACTIVEEVENT,
       SDL_KEYDOWN,
       SDL_KEYUP,
       SDL_MOUSEMOTION,
       SDL_MOUSEBUTTONDOWN,
       SDL_MOUSEBUTTONUP,
       SDL_JOYAXISMOTION,
       SDL_JOYBALLMOTION,
       SDL_JOYHATMOTION,
       SDL_JOYBUTTONDOWN,
       SDL_JOYBUTTONUP,
       SDL_QUIT,
       SDL_SYSWMEVENT,
       SDL_EVENT_RESERVEDA,
       SDL_EVENT_RESERVEDB,
       SDL_VIDEORESIZE,
       SDL_VIDEOEXPOSE,
       SDL_EVENT_RESERVED2,
       SDL_EVENT_RESERVED3,
       SDL_EVENT_RESERVED4,
       SDL_EVENT_RESERVED5,
       SDL_EVENT_RESERVED6,
       SDL_EVENT_RESERVED7,
       SDL_USEREVENT = 24,
       SDL_NUMEVENTS = 32
} SDL_EventType;

#define SDL_EVENTMASK(X)	(1<<(X))
typedef enum {
	SDL_ACTIVEEVENTMASK	= SDL_EVENTMASK(SDL_ACTIVEEVENT),
	SDL_KEYDOWNMASK		= SDL_EVENTMASK(SDL_KEYDOWN),
	SDL_KEYUPMASK		= SDL_EVENTMASK(SDL_KEYUP),
	SDL_KEYEVENTMASK	= SDL_EVENTMASK(SDL_KEYDOWN)|
	                          SDL_EVENTMASK(SDL_KEYUP),
	SDL_MOUSEMOTIONMASK	= SDL_EVENTMASK(SDL_MOUSEMOTION),
	SDL_MOUSEBUTTONDOWNMASK	= SDL_EVENTMASK(SDL_MOUSEBUTTONDOWN),
	SDL_MOUSEBUTTONUPMASK	= SDL_EVENTMASK(SDL_MOUSEBUTTONUP),
	SDL_MOUSEEVENTMASK	= SDL_EVENTMASK(SDL_MOUSEMOTION)|
	                          SDL_EVENTMASK(SDL_MOUSEBUTTONDOWN)|
	                          SDL_EVENTMASK(SDL_MOUSEBUTTONUP),
	SDL_JOYAXISMOTIONMASK	= SDL_EVENTMASK(SDL_JOYAXISMOTION),
	SDL_JOYBALLMOTIONMASK	= SDL_EVENTMASK(SDL_JOYBALLMOTION),
	SDL_JOYHATMOTIONMASK	= SDL_EVENTMASK(SDL_JOYHATMOTION),
	SDL_JOYBUTTONDOWNMASK	= SDL_EVENTMASK(SDL_JOYBUTTONDOWN),
	SDL_JOYBUTTONUPMASK	= SDL_EVENTMASK(SDL_JOYBUTTONUP),
	SDL_JOYEVENTMASK	= SDL_EVENTMASK(SDL_JOYAXISMOTION)|
	                          SDL_EVENTMASK(SDL_JOYBALLMOTION)|
	                          SDL_EVENTMASK(SDL_JOYHATMOTION)|
	                          SDL_EVENTMASK(SDL_JOYBUTTONDOWN)|
	                          SDL_EVENTMASK(SDL_JOYBUTTONUP),
	SDL_VIDEORESIZEMASK	= SDL_EVENTMASK(SDL_VIDEORESIZE),
	SDL_VIDEOEXPOSEMASK	= SDL_EVENTMASK(SDL_VIDEOEXPOSE),
	SDL_QUITMASK		= SDL_EVENTMASK(SDL_QUIT),
	SDL_SYSWMEVENTMASK	= SDL_EVENTMASK(SDL_SYSWMEVENT)
} SDL_EventMask ;
#define SDL_ALLEVENTS		0xFFFFFFFF

typedef struct SDL_ActiveEvent {
	Uint8 type;
	Uint8 gain;
	Uint8 state;
} SDL_ActiveEvent;

typedef struct SDL_KeyboardEvent {
	Uint8 type;	
	Uint8 which;
	Uint8 state;
	SDL_keysym keysym;
} SDL_KeyboardEvent;

typedef struct SDL_MouseMotionEvent {
	Uint8 type;
	Uint8 which;
	Uint8 state;
	Uint16 x, y;
	Sint16 xrel;
	Sint16 yrel;
} SDL_MouseMotionEvent;

typedef struct SDL_MouseButtonEvent {
	Uint8 type;
	Uint8 which;
	Uint8 button;
	Uint8 state;
	Uint16 x, y;
} SDL_MouseButtonEvent;

typedef struct SDL_JoyAxisEvent {
	Uint8 type;
	Uint8 which;
	Uint8 axis;
	Sint16 value;
} SDL_JoyAxisEvent;

typedef struct SDL_JoyBallEvent {
	Uint8 type;
	Uint8 which;
	Uint8 ball;
	Sint16 xrel;
	Sint16 yrel;
} SDL_JoyBallEvent;

typedef struct SDL_JoyHatEvent {
	Uint8 type;
	Uint8 which;
	Uint8 hat;
	Uint8 value;
} SDL_JoyHatEvent;

typedef struct SDL_JoyButtonEvent {
	Uint8 type;
	Uint8 which;
	Uint8 button;
	Uint8 state;
} SDL_JoyButtonEvent;

typedef struct SDL_ResizeEvent {
	Uint8 type;
	int w;
	int h;
} SDL_ResizeEvent;

typedef struct SDL_ExposeEvent {
	Uint8 type;
} SDL_ExposeEvent;

typedef struct SDL_QuitEvent {
	Uint8 type;
} SDL_QuitEvent;

typedef struct SDL_UserEvent {
	Uint8 type;
	int code;
	void *data1;
	void *data2;
} SDL_UserEvent;

struct SDL_SysWMmsg;
typedef struct SDL_SysWMmsg SDL_SysWMmsg;
typedef struct SDL_SysWMEvent {
	Uint8 type;
	SDL_SysWMmsg *msg;
} SDL_SysWMEvent;

typedef union SDL_Event {
	Uint8 type;
	SDL_ActiveEvent active;
	SDL_KeyboardEvent key;
	SDL_MouseMotionEvent motion;
	SDL_MouseButtonEvent button;
	SDL_JoyAxisEvent jaxis;
	SDL_JoyBallEvent jball;
	SDL_JoyHatEvent jhat;
	SDL_JoyButtonEvent jbutton;
	SDL_ResizeEvent resize;
	SDL_ExposeEvent expose;
	SDL_QuitEvent quit;
	SDL_UserEvent user;
	SDL_SysWMEvent syswm;
} SDL_Event;

typedef enum {
	SDL_ADDEVENT,
	SDL_PEEKEVENT,
	SDL_GETEVENT
} SDL_eventaction;

typedef int (SDLCALL *SDL_EventFilter)(const SDL_Event *event);

#define SDL_QUERY	-1
#define SDL_IGNORE	 0
#define SDL_DISABLE	 0
#define SDL_ENABLE	 1

/* SDL_timer.h defines */
#define SDL_TIMESLICE		10
#define TIMER_RESOLUTION	10
typedef Uint32 (SDLCALL *SDL_TimerCallback)(Uint32 interval);
typedef Uint32 (SDLCALL *SDL_NewTimerCallback)(Uint32 interval, void *param);
typedef struct _SDL_TimerID *SDL_TimerID;

/* SDL_version.h defines */
#define SDL_MAJOR_VERSION	1
#define SDL_MINOR_VERSION	2
#define SDL_PATCHLEVEL		15 

typedef struct SDL_version {
	Uint8 major;
	Uint8 minor;
	Uint8 patch;
} SDL_version;

/* SDL.h functions */
NV_SDL_FUN( int, SDL_Init, (Uint32 flags) );
NV_SDL_FUN( int, SDL_InitSubSystem, (Uint32 flags) );
NV_SDL_FUN( void, SDL_QuitSubSystem, (Uint32 flags) );
NV_SDL_FUN( Uint32, SDL_WasInit, (Uint32 flags) );
NV_SDL_FUN( void, SDL_Quit, (void) );

/* SDL_rwops.h functions */
NV_SDL_FUN( SDL_RWops *, SDL_RWFromFile, (const char *file, const char *mode) );
NV_SDL_FUN( SDL_RWops *, SDL_RWFromFP, (FILE *fp, int autoclose) );
NV_SDL_FUN( SDL_RWops *, SDL_RWFromMem, (void *mem, int size) );
NV_SDL_FUN( SDL_RWops *, SDL_RWFromConstMem, (const void *mem, int size) );
NV_SDL_FUN( SDL_RWops *, SDL_AllocRW, (void) );
NV_SDL_FUN( void, SDL_FreeRW, (SDL_RWops *area) );
NV_SDL_FUN( Uint16, SDL_ReadLE16, (SDL_RWops *src) );
NV_SDL_FUN( Uint16, SDL_ReadBE16, (SDL_RWops *src) );
NV_SDL_FUN( Uint32, SDL_ReadLE32, (SDL_RWops *src) );
NV_SDL_FUN( Uint32, SDL_ReadBE32, (SDL_RWops *src) );
NV_SDL_FUN( Uint64, SDL_ReadLE64, (SDL_RWops *src) );
NV_SDL_FUN( Uint64, SDL_ReadBE64, (SDL_RWops *src) );
NV_SDL_FUN( int, SDL_WriteLE16, (SDL_RWops *dst, Uint16 value) );
NV_SDL_FUN( int, SDL_WriteBE16, (SDL_RWops *dst, Uint16 value) );
NV_SDL_FUN( int, SDL_WriteLE32, (SDL_RWops *dst, Uint32 value) );
NV_SDL_FUN( int, SDL_WriteBE32, (SDL_RWops *dst, Uint32 value) );
NV_SDL_FUN( int, SDL_WriteLE64, (SDL_RWops *dst, Uint64 value) );
NV_SDL_FUN( int, SDL_WriteBE64, (SDL_RWops *dst, Uint64 value) );

/* SDL_video.h defines */
NV_SDL_FUN( int, SDL_VideoInit, (const char *driver_name, Uint32 flags) );
NV_SDL_FUN( void, SDL_VideoQuit, (void) );
NV_SDL_FUN( char *, SDL_VideoDriverName, (char *namebuf, int maxlen) );
NV_SDL_FUN( SDL_Surface *, SDL_GetVideoSurface, (void) );
NV_SDL_FUN( const SDL_VideoInfo *, SDL_GetVideoInfo, (void) );
NV_SDL_FUN( int, SDL_VideoModeOK, (int width, int height, int bpp, Uint32 flags) );
NV_SDL_FUN( SDL_Rect **, SDL_ListModes, (SDL_PixelFormat *format, Uint32 flags) );
NV_SDL_FUN( SDL_Surface *, SDL_SetVideoMode, (int width, int height, int bpp, Uint32 flags) );
NV_SDL_FUN( void, SDL_UpdateRects, (SDL_Surface *screen, int numrects, SDL_Rect *rects) );
NV_SDL_FUN( void, SDL_UpdateRect, (SDL_Surface *screen, Sint32 x, Sint32 y, Uint32 w, Uint32 h) );
NV_SDL_FUN( int, SDL_Flip, (SDL_Surface *screen) );
NV_SDL_FUN( int, SDL_SetGamma, (float red, float green, float blue) );
NV_SDL_FUN( int, SDL_SetGammaRamp, (const Uint16 *red, const Uint16 *green, const Uint16 *blue) );
NV_SDL_FUN( int, SDL_GetGammaRamp, (Uint16 *red, Uint16 *green, Uint16 *blue) );
NV_SDL_FUN( int, SDL_SetColors, (SDL_Surface *surface, SDL_Color *colors, int firstcolor, int ncolors) );
NV_SDL_FUN( int, SDL_SetPalette, (SDL_Surface *surface, int flags, SDL_Color *colors, int firstcolor, int ncolors) );
NV_SDL_FUN( Uint32, SDL_MapRGB, (const SDL_PixelFormat * const format, const Uint8 r, const Uint8 g, const Uint8 b) );
NV_SDL_FUN( Uint32, SDL_MapRGBA, (const SDL_PixelFormat * const format, const Uint8 r, const Uint8 g, const Uint8 b, const Uint8 a) );
NV_SDL_FUN( void, SDL_GetRGB, (Uint32 pixel, const SDL_PixelFormat * const fmt, Uint8 *r, Uint8 *g, Uint8 *b) );
NV_SDL_FUN( void, SDL_GetRGBA, (Uint32 pixel,	const SDL_PixelFormat * const fmt, Uint8 *r, Uint8 *g, Uint8 *b, Uint8 *a) );
NV_SDL_FUN( SDL_Surface *, SDL_CreateRGBSurface, (Uint32 flags, int width, int height, int depth, Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask) );
NV_SDL_FUN( SDL_Surface *, SDL_CreateRGBSurfaceFrom, (void *pixels, int width, int height, int depth, int pitch, Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask) );
NV_SDL_FUN( void, SDL_FreeSurface, (SDL_Surface *surface) );
NV_SDL_FUN( int, SDL_LockSurface, (SDL_Surface *surface) );
NV_SDL_FUN( void, SDL_UnlockSurface, (SDL_Surface *surface) );
NV_SDL_FUN( SDL_Surface *, SDL_LoadBMP_RW, (SDL_RWops *src, int freesrc) );
NV_SDL_FUN( int, SDL_SaveBMP_RW, (SDL_Surface *surface, SDL_RWops *dst, int freedst) );
NV_SDL_FUN( int, SDL_SetColorKey, (SDL_Surface *surface, Uint32 flag, Uint32 key) );
NV_SDL_FUN( int, SDL_SetAlpha, (SDL_Surface *surface, Uint32 flag, Uint8 alpha) );
NV_SDL_FUN( SDL_bool, SDL_SetClipRect, (SDL_Surface *surface, const SDL_Rect *rect) );
NV_SDL_FUN( void, SDL_GetClipRect, (SDL_Surface *surface, SDL_Rect *rect) );
NV_SDL_FUN( SDL_Surface *, SDL_ConvertSurface, (SDL_Surface *src, SDL_PixelFormat *fmt, Uint32 flags) );
NV_SDL_FUN( int, SDL_UpperBlit, (SDL_Surface *src, SDL_Rect *srcrect, SDL_Surface *dst, SDL_Rect *dstrect) );
NV_SDL_FUN( int, SDL_LowerBlit, (SDL_Surface *src, SDL_Rect *srcrect, SDL_Surface *dst, SDL_Rect *dstrect) );
NV_SDL_FUN( int, SDL_FillRect, (SDL_Surface *dst, SDL_Rect *dstrect, Uint32 color) );
NV_SDL_FUN( SDL_Surface *, SDL_DisplayFormat, (SDL_Surface *surface) );
NV_SDL_FUN( SDL_Surface *, SDL_DisplayFormatAlpha, (SDL_Surface *surface) );
NV_SDL_FUN( SDL_Overlay *, SDL_CreateYUVOverlay, (int width, int height, Uint32 format, SDL_Surface *display) );
NV_SDL_FUN( int, SDL_LockYUVOverlay, (SDL_Overlay *overlay) );
NV_SDL_FUN( void, SDL_UnlockYUVOverlay, (SDL_Overlay *overlay) );
NV_SDL_FUN( int, SDL_DisplayYUVOverlay, (SDL_Overlay *overlay, SDL_Rect *dstrect) );
NV_SDL_FUN( void, SDL_FreeYUVOverlay, (SDL_Overlay *overlay) );
NV_SDL_FUN( int, SDL_GL_LoadLibrary, (const char *path) );
NV_SDL_FUN( void *, SDL_GL_GetProcAddress, (const char* proc) );
NV_SDL_FUN( int, SDL_GL_SetAttribute, (SDL_GLattr attr, int value) );
NV_SDL_FUN( int, SDL_GL_GetAttribute, (SDL_GLattr attr, int* value) );
NV_SDL_FUN( void, SDL_GL_SwapBuffers, (void) );
NV_SDL_FUN( void, SDL_GL_UpdateRects, (int numrects, SDL_Rect* rects) );
NV_SDL_FUN( void, SDL_GL_Lock, (void) );
NV_SDL_FUN( void, SDL_GL_Unlock, (void) );
NV_SDL_FUN( void, SDL_WM_SetCaption, (const char *title, const char *icon) );
NV_SDL_FUN( void, SDL_WM_GetCaption, (char **title, char **icon) );
NV_SDL_FUN( void, SDL_WM_SetIcon, (SDL_Surface *icon, Uint8 *mask) );
NV_SDL_FUN( int, SDL_WM_IconifyWindow, (void) );
NV_SDL_FUN( int, SDL_WM_ToggleFullScreen, (SDL_Surface *surface) );
NV_SDL_FUN( SDL_GrabMode, SDL_WM_GrabInput, (SDL_GrabMode mode) ); 

/* SDL_audio.h functions */
NV_SDL_FUN( int, SDL_AudioInit, (const char *driver_name) );
NV_SDL_FUN( void, SDL_AudioQuit, (void) ); 
NV_SDL_FUN( char *, SDL_AudioDriverName, (char *namebuf, int maxlen) );
NV_SDL_FUN( int, SDL_OpenAudio, (SDL_AudioSpec *desired, SDL_AudioSpec *obtained) );
NV_SDL_FUN( SDL_audiostatus, SDL_GetAudioStatus, (void) );
NV_SDL_FUN( void, SDL_PauseAudio, (int pause_on) );
NV_SDL_FUN( SDL_AudioSpec *, SDL_LoadWAV_RW, (SDL_RWops *src, int freesrc, SDL_AudioSpec *spec, Uint8 **audio_buf, Uint32 *audio_len) );
NV_SDL_FUN( void, SDL_FreeWAV, (Uint8 *audio_buf) );
NV_SDL_FUN( int, SDL_BuildAudioCVT, (SDL_AudioCVT *cvt, Uint16 src_format, Uint8 src_channels, int src_rate, Uint16 dst_format, Uint8 dst_channels, int dst_rate) );
NV_SDL_FUN( int, SDL_ConvertAudio, (SDL_AudioCVT *cvt) );
NV_SDL_FUN( void, SDL_MixAudio, (Uint8 *dst, const Uint8 *src, Uint32 len, int volume) );
NV_SDL_FUN( void, SDL_LockAudio, (void) );
NV_SDL_FUN( void, SDL_UnlockAudio, (void) );
NV_SDL_FUN( void, SDL_CloseAudio, (void) );  

/* SDL_cpuinfo.h functions */
NV_SDL_FUN( SDL_bool, SDL_HasRDTSC, (void) );
NV_SDL_FUN( SDL_bool, SDL_HasMMX, (void) );
NV_SDL_FUN( SDL_bool, SDL_HasMMXExt, (void) );
NV_SDL_FUN( SDL_bool, SDL_Has3DNow, (void) );
NV_SDL_FUN( SDL_bool, SDL_Has3DNowExt, (void) );
NV_SDL_FUN( SDL_bool, SDL_HasSSE, (void) );
NV_SDL_FUN( SDL_bool, SDL_HasSSE2, (void) );
NV_SDL_FUN( SDL_bool, SDL_HasAltiVec, (void) ); 

/* SDL_error.h functions */
NV_SDL_FUN( void, SDL_SetError, (const char *fmt, ...) );
NV_SDL_FUN( char *, SDL_GetError, (void) );
NV_SDL_FUN( void, SDL_ClearError, (void) );
NV_SDL_FUN( void, SDL_Error, (SDL_errorcode code) );

/* SDL_active.h functions */
NV_SDL_FUN( Uint8, SDL_GetAppState, (void) );

/* SDL_keyboard.h functions */
NV_SDL_FUN( int, SDL_EnableUNICODE, (int enable) );
NV_SDL_FUN( int, SDL_EnableKeyRepeat, (int delay, int interval) );
NV_SDL_FUN( void, SDL_GetKeyRepeat, (int *delay, int *interval) );
NV_SDL_FUN( Uint8 *, SDL_GetKeyState, (int *numkeys) );
NV_SDL_FUN( SDLMod, SDL_GetModState, (void) );
NV_SDL_FUN( void, SDL_SetModState, (SDLMod modstate) );
NV_SDL_FUN( char *, SDL_GetKeyName, (SDLKey key) );
 
/* SDL_mouse.h functions */
NV_SDL_FUN( Uint8, SDL_GetMouseState, (int *x, int *y) );
NV_SDL_FUN( Uint8, SDL_GetRelativeMouseState, (int *x, int *y) );
NV_SDL_FUN( void, SDL_WarpMouse, (Uint16 x, Uint16 y) );
NV_SDL_FUN( SDL_Cursor *, SDL_CreateCursor, (Uint8 *data, Uint8 *mask, int w, int h, int hot_x, int hot_y) );
NV_SDL_FUN( void, SDL_SetCursor, (SDL_Cursor *cursor) );
NV_SDL_FUN( SDL_Cursor *, SDL_GetCursor, (void) );
NV_SDL_FUN( void, SDL_FreeCursor, (SDL_Cursor *cursor) );
NV_SDL_FUN( int, SDL_ShowCursor, (int toggle) );

/* SDL_joystick.h functions */
NV_SDL_FUN( int, SDL_NumJoysticks, (void) );
NV_SDL_FUN( const char *, SDL_JoystickName, (int device_index) );
NV_SDL_FUN( SDL_Joystick *, SDL_JoystickOpen, (int device_index) );
NV_SDL_FUN( int, SDL_JoystickOpened, (int device_index) );
NV_SDL_FUN( int, SDL_JoystickIndex, (SDL_Joystick *joystick) );
NV_SDL_FUN( int, SDL_JoystickNumAxes, (SDL_Joystick *joystick) );
NV_SDL_FUN( int, SDL_JoystickNumBalls, (SDL_Joystick *joystick) );
NV_SDL_FUN( int, SDL_JoystickNumHats, (SDL_Joystick *joystick) );
NV_SDL_FUN( int, SDL_JoystickNumButtons, (SDL_Joystick *joystick) );
NV_SDL_FUN( void, SDL_JoystickUpdate, (void) );
NV_SDL_FUN( int, SDL_JoystickEventState, (int state) );
NV_SDL_FUN( Sint16, SDL_JoystickGetAxis, (SDL_Joystick *joystick, int axis) );
NV_SDL_FUN( Uint8, SDL_JoystickGetHat, (SDL_Joystick *joystick, int hat) );
NV_SDL_FUN( int, SDL_JoystickGetBall, (SDL_Joystick *joystick, int ball, int *dx, int *dy) );
NV_SDL_FUN( Uint8, SDL_JoystickGetButton, (SDL_Joystick *joystick, int button) );
NV_SDL_FUN( void, SDL_JoystickClose, (SDL_Joystick *joystick) );

/* SDL_timer.h functions */
NV_SDL_FUN( Uint32, SDL_GetTicks, (void) );
NV_SDL_FUN( void, SDL_Delay, (Uint32 ms) );
NV_SDL_FUN( int, SDL_SetTimer, (Uint32 interval, SDL_TimerCallback callback) );
NV_SDL_FUN( SDL_TimerID, SDL_AddTimer, (Uint32 interval, SDL_NewTimerCallback callback, void *param) );
NV_SDL_FUN( SDL_bool, SDL_RemoveTimer, (SDL_TimerID t) );

/* SDL_version.h functions */
NV_SDL_FUN( const SDL_version *, SDL_Linked_Version, (void) ); 

/* SDL_event.h functions */
NV_SDL_FUN( void, SDL_PumpEvents, (void) );
NV_SDL_FUN( int, SDL_PeepEvents, (SDL_Event *events, int numevents, SDL_eventaction action, Uint32 mask) );
NV_SDL_FUN( int, SDL_PollEvent, (SDL_Event *event) );
NV_SDL_FUN( int, SDL_WaitEvent, (SDL_Event *event) );
NV_SDL_FUN( int, SDL_PushEvent, (SDL_Event *event) );
NV_SDL_FUN( void, SDL_SetEventFilter, (SDL_EventFilter filter) );
NV_SDL_FUN( SDL_EventFilter, SDL_GetEventFilter, (void) );
NV_SDL_FUN( Uint8, SDL_EventState, (Uint8 etype, int state) );

#define SDL_RWseek(ctx, offset, whence)	(ctx)->seek(ctx, offset, whence)
#define SDL_RWtell(ctx)			(ctx)->seek(ctx, 0, RW_SEEK_CUR)
#define SDL_RWread(ctx, ptr, size, n)	(ctx)->read(ctx, ptr, size, n)
#define SDL_RWwrite(ctx, ptr, size, n)	(ctx)->write(ctx, ptr, size, n)
#define SDL_RWclose(ctx)		(ctx)->close(ctx)
#define SDL_BlitSurface SDL_UpperBlit
#define SDL_VERSION(X)							\
{									\
	(X)->major = SDL_MAJOR_VERSION;					\
	(X)->minor = SDL_MINOR_VERSION;					\
	(X)->patch = SDL_PATCHLEVEL;					\
}
#define SDL_VERSIONNUM(X, Y, Z)						\
	((X)*1000 + (Y)*100 + (Z))
#define SDL_COMPILEDVERSION \
	SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL)
#define SDL_VERSION_ATLEAST(X, Y, Z) \
	(SDL_COMPILEDVERSION >= SDL_VERSIONNUM(X, Y, Z))
 
}

namespace nv {
/* Dynamic load support */
#	if defined( NV_SDL_DYNAMIC )
		bool load_sdl_library( const char* path = NV_SDL_PATH );
#	else
		inline bool load_sdl_library( const char* path = "" ) { return true; }
#	endif
}

#endif // NV_LIB_SDL_HH
