// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_IMAGE_HH
#define NV_IMAGE_HH

#include <nv/common.hh>
#include <glm/glm.hpp>

namespace nv
{
	/**
	 * Defines the starting position of some region and its size.
	 */
	struct region
	{
		glm::ivec2 pos;
		glm::ivec2 size;

		region() : pos(0,0), size(0,0) {}
		region( glm::ivec2 pos, glm::ivec2 size ) : pos(pos), size(size) {}
	};

	/**
	 * Holder for images.
	 *
	 * By an image we understand a group of pixels, limited to X and Y
	 * dimensions, of a certain depth (by depth we mean how rich
	 * the colours are.
	 */
	class image
	{
	public:
		/**
		 * Constructor
		 *
		 * @arg[in] size : Dimensions of the image as a vector.
		 * @arg[in] depth :  Depth of the image.
		 */
		image( glm::ivec2 size, size_t depth );
		/**
		 * Full constructor.
		 *
		 * Allows creation of an image with specified data. It is
		 * to reverse the image (from left to right).
		 *
		 * @arg[in] size : Dimensions of the image as a vector.
		 * @arg[in] depth : Depth of the image.
		 * @arg[in] data : Data of the image.
		 * @arg[in] reversed : Determines if the image should be
		 * 	reversed or not.
		 */
		image( glm::ivec2 size, size_t depth, const uint8 * data, bool reversed = false );
		/**
		 * Fills the image with a given value.
		 *
		 * @arg[in] value : Value to fill.
		 */
		void fill( uint8 value );
		/**
		 * Just like fill, but a region can be specified.
		 *
		 * @arg[in] r : Region to be filled.
		 * @arg[in] data : Data to fill the region
		 * @arg[in] stride : 
		 */
		void set_region( region r, const uint8 * data, size_t stride = 0 );
		/**
		 * Default destructor. Deallocates data.
		 */
		virtual ~image();
		/**
		 * Getter for data.
		 *
		 * @return The data.
		 */
		const uint8 * get_data() const { return m_data; }
		/**
		 * Getter for size.
		 *
		 * @return Returns the size of the image as a vector.
		 */
		const glm::ivec2 get_size() const { return m_size; }
		/**
		 * Getter for depth.
		 *
		 * @return Returns depth of the image.
		 */
		const size_t get_depth() const { return m_depth; }
	protected:
		/** Defines the size of the image as a vector. */
		glm::ivec2 m_size;
		/** Deefines the depth of the image. */
		size_t     m_depth;
		/** Holder for data. */
		uint8*     m_data;
	};

}

#endif // NV_IMAGE_HH

