// Copyright (C) 2011 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/time.hh"

#include "nv/logging.hh"

#if NV_COMPILER == NV_MSVC
#define WIN32_LEAN_AND_MEAN 
#include <windows.h>
#include <intrin.h>
#pragma intrinsic(__rdtsc)
#else
#include <unistd.h>
#include <sys/time.h>
#endif

#include <ctime>

struct timer_impl
{
	timer_impl()
	{
		clock_zero = clock();
#if NV_COMPILER == NV_MSVC
		QueryPerformanceFrequency(&frequency);
		QueryPerformanceCounter(&query_zero);
#else
		gettimeofday(&timeval_zero, NULL);
#endif
	}
	clock_t clock_zero;
#if NV_COMPILER == NV_MSVC
	LARGE_INTEGER query_zero;
	LARGE_INTEGER frequency;
#else
	struct timeval timeval_zero;
#endif
};

static timer_impl zero_timer;

volatile nv::uint64 nv::get_ticks()
{
#if NV_COMPILER == NV_MSVC
	return __rdtsc();
#elif NV_COMPILER == NV_GNUC
	register long long ticks asm("eax");
	asm volatile (".byte 15, 49" : : : "eax", "edx");
	return ticks;
#else
	return 0; // unsupported
#endif
}

void nv::sleep( uint32 ms )
{
#if NV_COMPILER == NV_MSVC
	Sleep( ms );
#elif NV_COMPILER == NV_GNUC
	usleep( ms * 1000 );
#else
#endif
}

nv::uint32 nv::get_cpu_ms()
{
	return (uint32)( (f32)( clock() - zero_timer.clock_zero ) / ( (f32)CLOCKS_PER_SEC / 1000.0 ) ) ;
}

nv::uint32 nv::get_cpu_us()
{
	return (uint32)( (f32)( clock() - zero_timer.clock_zero ) / ( (f32)CLOCKS_PER_SEC / 1000000.0 ) ) ;
}

nv::uint32 nv::get_system_ms()
{
#if NV_COMPILER == NV_MSVC
	LARGE_INTEGER now;
	QueryPerformanceCounter(&now);
	LONGLONG result = now.QuadPart - zero_timer.query_zero.QuadPart;
	return (uint32) (1000.0 * result / zero_timer.frequency.QuadPart);
#else
	struct timeval now;
	gettimeofday(&now, NULL);
	return (now.tv_sec - zero_timer.timeval_zero.tv_sec)*1000+(now.tv_usec-zero_timer.timeval_zero.tv_usec)/1000;
#endif
}

nv::uint32 nv::get_system_us()
{
#if NV_COMPILER == NV_MSVC
	LARGE_INTEGER now;
	QueryPerformanceCounter(&now);
	LONGLONG result = now.QuadPart - zero_timer.query_zero.QuadPart;
	return (uint32) (1000000.0 * result / zero_timer.frequency.QuadPart);
#else
	struct timeval now;
	gettimeofday(&now, NULL);
	return (now.tv_sec - zero_timer.timeval_zero.tv_sec)*1000000+(now.tv_usec - zero_timer.timeval_zero.tv_usec);
#endif
}
