// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file logger.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief logger implementation
 */

#ifndef NV_LOGGER_HH
#define NV_LOGGER_HH

#include <string>
#include <iosfwd>
#include <list>
#include <nv/logging.hh>

namespace nv
{
	/**
	 * Class representing log output.
	 *
	 * Base class for all logging output targets.
	 */
	class log_sink
	{
	public:
		/**
		 * Logging function.
		 *
		 * @param level level on which to log.
		 * @param message message to be logged.
		 */
		virtual void log( log_level level, const std::string& message ) = 0;
		/**
         * Optional timestamp string
		 */
		const char* timestamp() const;
		/**
		 * Enforcement of virtual destructor.
		 */
		virtual ~log_sink() {};
	};

	/**
	 * Logger class.
	 *
	 * Allows to catch all logging in ORE and in the application using ORE
	 * logging macros. Initially does nothing, needs to have some log sinks
	 * added.
	 */
	class logger : public logger_base
	{
	public:
		/**
		 * Constructor, taking the minimum level for logging.
		 *
		 * No matter what level sinks you add, the passed level is the
		 * overriding cutoff.
		 */
		explicit logger( int llevel )
		{
			level = llevel;
		}

		/**
		 * Logging function.
		 *
		 * Goes over all registered log sinks and prints to them depending
		 * on the log level of the sink. Messages that are lower level than
		 * the initial level passed to the constructor never even reach this
		 * function.
		 *
		 * @param level level on which to log.
		 * @param message message to be logged.
		 */
		virtual void log( log_level level, const std::string& message );

		/** 
		 * Log sink registering.
		 * 
		 * Registers a new log sink. Ownership is passed to logger - log sink
		 * will be disposed at program shutdown.
		 *
		 * @param sink log sink to be registered.
		 * @param level level on which the sink should log.
		 */
		virtual void add_sink( log_sink* sink, int level );

		/** 
		 * Log sink removal.
		 * 
		 * Removes a log sink. If log sink doesn't exist, nothing will 
		 * happen.
		 *
		 * @param sink log sink to be removed.
		 * @returns true if sink was found, false otherwise
		 */
		virtual bool remove_sink( log_sink* sink );

		/**
		 * Destructor.
		 * 
		 * Also destroys all file sinks.
		 */
		virtual ~logger();

	protected:
		/** Type for the log sink list. */
		typedef std::list< std::pair< log_level, log_sink* > > log_sink_list;

		/** Log sink list. */
		log_sink_list m_log_sinks;
	};

	/**
	 * Console logger sink.
	 *
	 * Logs to std::out -- be sure a console window is open, or std::out redirected!
	 */
	class log_console_sink : public log_sink
	{
	public:
		/**
		 * Log sink constructor
		 */
		log_console_sink( bool coloring = true );
		/**
		 * Logging function.
		 */
		virtual void log( log_level level, const std::string& message );

	private:
		void* m_handle;
		bool  m_color;
	};

	/**
	 * General stream sink.
	 *
	 * Logs to passed stream.
	 */
	class log_stream_sink : public log_sink
	{
	public:
		/**
		 * Constructor.
		 *
		 * Logs to the passed stream. The stream is NOT disposed at destruction.
		 * Flushing can be controlled by optional flush parameter.
		 *
		 * @param stream stream to be logged to.
		 * @param flush_always if set to false, wont flush after each line.
		 */
		log_stream_sink( std::ostream* stream, bool flush_always ) 
			: m_stream(stream), m_flush(flush_always) {}

		/**
		 * Logging function.
		 */
		virtual void log( log_level level, const std::string& message );

	protected:
		/** Stored stream. */
		std::ostream* m_stream;
		/** Controls flushing. */
		bool m_flush;
	};

	/**
	 * File logger sink.
	 *
	 * Logs to std::out -- be sure a console window is open, or std::out redirected!
	 */
	class log_file_sink : public log_stream_sink
	{
	public:
		/**
		 * Constructor.
		 *
		 * Logs to the file passed. File is closed and disposed of after ending.
		 * File is not appended, it's overwritten. If file is not creatable,
		 * the constructor will throw!
		 *
		 * @param file_name file to be logged to.
		 * @param flush_always if set to false, wont flush after each line.
		 * @throws runtime_error if file_name cannot be opened.
		 */
		log_file_sink( const std::string file_name, bool flush_always = true );

		/**
		 * Destructor.
		 *
		 * Flushes, closes file and disposes of the stream.
		 */
		virtual ~log_file_sink();
	};

}

#endif // NV_LOGGER_HH
