// Copyright (C) 2012 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_TYPES_HH
#define NV_TYPES_HH

#include <glm/glm.hpp>
#include <nv/common.hh>
#include <nv/object.hh>
#include <type_traits>
#include <utility>
#include <unordered_map>
#include <vector>
#include <string>

#define NV_REGISTER_NAME( s ) template <> inline const char* nv::get_type_name<s>   () { return #s; }

namespace nv
{

	enum type
	{
		INT,
		BYTE,
		SHORT,
		UINT,
		UBYTE,
		USHORT,
		FLOAT,
		FLOAT_VECTOR_2,
		FLOAT_VECTOR_3,
		FLOAT_VECTOR_4,
		FLOAT_MATRIX_2,
		FLOAT_MATRIX_3,
		FLOAT_MATRIX_4,
		INT_VECTOR_2,
		INT_VECTOR_3,
		INT_VECTOR_4
	};

	typedef glm::vec2 vec2;
	typedef glm::vec3 vec3;
	typedef glm::vec4 vec4;

	typedef glm::ivec2 ivec2;
	typedef glm::ivec3 ivec3;
	typedef glm::ivec4 ivec4;

	typedef glm::mat2 mat2;
	typedef glm::mat3 mat3;
	typedef glm::mat4 mat4;

	template < type EnumType > struct enum_to_type {};

	template <> struct enum_to_type< INT >   { typedef int type; };
	template <> struct enum_to_type< UINT >  { typedef unsigned int type; };
	template <> struct enum_to_type< SHORT > { typedef short type; };
	template <> struct enum_to_type< USHORT >{ typedef unsigned short type; };
	template <> struct enum_to_type< BYTE >  { typedef char type; };
	template <> struct enum_to_type< UBYTE > { typedef unsigned char type; };
	template <> struct enum_to_type< FLOAT > { typedef f32 type; };

	template <> struct enum_to_type< FLOAT_VECTOR_2 > { typedef vec2 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_3 > { typedef vec3 type; };
	template <> struct enum_to_type< FLOAT_VECTOR_4 > { typedef vec4 type; };

	template <> struct enum_to_type< INT_VECTOR_2 > { typedef ivec2 type; };
	template <> struct enum_to_type< INT_VECTOR_3 > { typedef ivec3 type; };
	template <> struct enum_to_type< INT_VECTOR_4 > { typedef ivec4 type; };

	template <> struct enum_to_type< FLOAT_MATRIX_2 > { typedef mat2 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_3 > { typedef mat3 type; };
	template <> struct enum_to_type< FLOAT_MATRIX_4 > { typedef mat4 type; };

	template < typename TYPE > struct type_to_enum {};

	template <> struct type_to_enum< int >           { static const type type = INT; };
	template <> struct type_to_enum< unsigned int >  { static const type type = UINT; };
	template <> struct type_to_enum< short >         { static const type type = SHORT; };
	template <> struct type_to_enum< unsigned short >{ static const type type = USHORT; };
	template <> struct type_to_enum< char >          { static const type type = BYTE; };
	template <> struct type_to_enum< unsigned char > { static const type type = UBYTE; };
	template <> struct type_to_enum< f32 > { static const type type = FLOAT; };

	template <> struct type_to_enum< vec2 > { static const type type = FLOAT_VECTOR_2; };
	template <> struct type_to_enum< vec3 > { static const type type = FLOAT_VECTOR_3; };
	template <> struct type_to_enum< vec4 > { static const type type = FLOAT_VECTOR_4; };

	template <> struct type_to_enum< ivec2 > { static const type type = INT_VECTOR_2; };
	template <> struct type_to_enum< ivec3 > { static const type type = INT_VECTOR_3; };
	template <> struct type_to_enum< ivec4 > { static const type type = INT_VECTOR_4; };

	template <> struct type_to_enum< mat2 > { static const type type = FLOAT_MATRIX_2; };
	template <> struct type_to_enum< mat3 > { static const type type = FLOAT_MATRIX_3; };
	template <> struct type_to_enum< mat4 > { static const type type = FLOAT_MATRIX_4; };

	template <typename TYPE>
    inline const char* get_type_name()
    {
        static_assert( false, "Type not implemented!" );
    }

	template <> inline const char* get_type_name<int>   () { return "sint"; }
    template <> inline const char* get_type_name<sint8> () { return "sint8"; }
    template <> inline const char* get_type_name<sint16>() { return "sint16"; }
    template <> inline const char* get_type_name<sint32>() { return "sint32"; }
    template <> inline const char* get_type_name<sint64>() { return "sint64"; }

	template <> inline const char* get_type_name<unsigned int>() { return "uint"; }
	template <> inline const char* get_type_name<uint8> ()       { return "uint8"; }
    template <> inline const char* get_type_name<uint16>()       { return "uint16"; }
    template <> inline const char* get_type_name<uint32>()       { return "uint32"; }
    template <> inline const char* get_type_name<uint64>()       { return "uint64"; }

	template <> inline const char* get_type_name<f32> () { return "f32"; }
	template <> inline const char* get_type_name<f64> () { return "f64"; }

	template <> inline const char* get_type_name< vec2 > () { return "vec2"; }
	template <> inline const char* get_type_name< vec3 > () { return "vec3"; }
	template <> inline const char* get_type_name< vec4 > () { return "vec4"; }

	template <> inline const char* get_type_name< ivec2 > () { return "ivec2"; }
	template <> inline const char* get_type_name< ivec3 > () { return "ivec3"; }
	template <> inline const char* get_type_name< ivec4 > () { return "ivec4"; }

	template <> inline const char* get_type_name< mat2 > () { return "mat2"; }
	template <> inline const char* get_type_name< mat3 > () { return "mat3"; }
	template <> inline const char* get_type_name< mat4 > () { return "mat4"; }

	template <> inline const char* get_type_name<bool> () { return "bool"; }

	template <> inline const char* get_type_name<std::string> () { return "string"; }
	template <> inline const char* get_type_name<object>      () { return "object"; }

	template<typename T>
	struct is_container
	{
	private:
		typedef char                      yes;
		typedef struct { char array[2]; } no;
		template<typename C> static yes test(typename C::iterator*);
		template<typename C> static no  test(...);
	public:
		static const bool value = sizeof(test<T>(0)) == sizeof(yes);
	};

	template<>
	struct is_container< std::string > {
		static const bool value = false;
	};


	template <typename TYPE>
    std::size_t get_type_id()
    {
        static std::size_t type_id = std::hash<std::string>()(std::string(get_type_name<TYPE>()));
        return type_id;
    };

	struct hash_string
    {
        std::size_t hash;
        const char* text;
		hash_string() {}
		hash_string( const char* a_text ) : 
			text( a_text ) 
		{
			hash = std::hash<std::string>()( std::string( a_text ) ); 
		}
		inline bool operator == (const hash_string& hs ) const
		{
			return hs.hash == hash;
		}
    };

}

namespace std {
	template<>
	struct hash<nv::hash_string> {
		size_t operator()(const nv::hash_string &hs) const 
		{
			return hs.hash;
		}
	};
}

namespace nv
{
	struct type_entry;

	enum type_flag
	{
		TF_POINTER      = 0x01, //< field is a pointer
		TF_NOSERIALIZE  = 0x02, //< ignore during serialization
		TF_INVISIBLE    = 0x04, //< field invisible to API
		TF_READONLY     = 0x08, //< read only field
		TF_SIMPLETYPE   = 0x10, //< raw binary I/O possible
		TF_OWNED        = 0x20,
		TF_CONTAINER    = 0x40, //< is a container
	};

	struct type_field
	{
		hash_string  name;      //< name of the field
		hash_string  type_name; //< name of the type of the field
		type_entry*  type;      //< pointer to field type
		unsigned int flags;     //< flags 
		size_t       offset;

		template< typename TOBJECT, typename TFIELD >
		type_field( hash_string name, TFIELD TOBJECT::*field, typename std::enable_if< is_container<TFIELD>::value, void* >::type = nullptr )
			: name(name)
			, type_name( get_type_name< std::remove_pointer<TFIELD::value_type>::type >() )
			, type( nullptr )
			, flags( 0 )
			, offset( offsetof( TOBJECT, *field ) )
			// NOTE: if offsetof behaves badly, check offset_of in common.hh
		{
			flags = TF_CONTAINER |
				( std::is_pointer<TFIELD::value_type>::value ? TF_POINTER : 0 ) |
				( std::is_pod<TFIELD::value_type>::value ? TF_SIMPLETYPE : 0 );
		}

		template< typename TOBJECT, typename TFIELD >
		type_field( hash_string name, TFIELD TOBJECT::*field, typename std::enable_if< !is_container<TFIELD>::value, void* >::type = nullptr )
			: name(name)
			, type_name( get_type_name< std::remove_pointer<TFIELD>::type >() )
			, type( nullptr )
			, flags( 0 )
			, offset( offsetof( TOBJECT, *field ) )
			// NOTE: if offsetof behaves badly, check offset_of in common.hh
		{
			flags = 
				( std::is_pointer<TFIELD>::value ? TF_POINTER : 0 ) |
				( std::is_pod<TFIELD>::value ? TF_SIMPLETYPE : 0 );
		}

		type_field& flag( unsigned int f )
		{
			flags |= f;
			return *this;
		}
	};

	struct type_enum
	{
		hash_string name;
		int         value;
		type_enum( hash_string name, int value ) : name(name), value(value) {}
	};

    struct type_entry
    {
		 // Function types for the constructor and destructor of registered types
	    typedef void (*constructor_func)(void*);
	    typedef void (*destructor_func)(void*);

		// Parent type database
        class type_database* type_db;

        // Scoped C++ name of the type
        hash_string name;
 
        // Pointers to the constructor and destructor functions
        constructor_func constructor;
        destructor_func  destructor;
 
        // Result of sizeof(type) operation
        size_t size;

		// Base type
		type_entry* base_type;

		// Field list
		std::vector<type_field> field_list;

		// Enum list
		std::vector<type_enum> enum_list;

		template <int TYPE>
		type_entry& base()
		{
			base_type = type_db->get_type( get_type_name<TYPE>() )
		}

		template <int SIZE>
		type_entry& fields( type_field (&init_fields)[SIZE] )
		{
			for (int i = 0; i < SIZE; i++)
			{
				type_field f = init_fields[i];
				f.type = type_db->get_type(f.type_name);
				field_list.push_back(f);
			}
			return *this;
		}

		template <int SIZE>
		type_entry& enums( type_enum (&init_enums)[SIZE] )
		{
			for (int i = 0; i < SIZE; i++)
			{
				enum_list.push_back( init_enums[i] )
			}
			return *this;
		}
    };

    class type_database
    {
    public:
		template< typename TYPE >
        type_entry& create_type()
		{
			hash_string name( get_type_name<TYPE>() );
			type_entry* i_type = nullptr;
			type_map::iterator it = m_types.find( name );
			if ( it != m_types.end() ) 
			{
				return *(it->second);
			}
			i_type          = new type_entry;
			i_type->type_db = this;
			i_type->name    = name;
			i_type->size    = sizeof(TYPE);
			
			i_type->constructor = ConstructObject<TYPE>;
			i_type->destructor  = DestructObject<TYPE>;

			m_types[name] = i_type;
			return *i_type;
		}

        type_entry* get_type( hash_string name )
		{
			type_map::iterator it = m_types.find( name );
			if ( it != m_types.end() ) 
			{
				return it->second;
			}
			return nullptr;
		}
    private:
        typedef std::unordered_map<hash_string, type_entry*> type_map;
        type_map m_types;
	};

	template <typename TYPE> void ConstructObject(void* object)
    {
        // Use placement new to call the constructor
        new (object) TYPE;
    }
    template <typename TYPE> void DestructObject(void* object)
    {
        // Explicit call of the destructor
        ((TYPE*)object)->TYPE::~TYPE();
    }

}

#endif NV_TYPES_HH