// Copyright (C) 2012-2013 ChaosForge / Kornel Kisielewicz
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_COMMON_HH
#define NV_COMMON_HH

#include <typeinfo>

// NV Library version
#define NV_VERSION_MAJOR 0
#define NV_VERSION_MINOR 1
#define NV_VERSION_PATCH 0
#define NV_VERSION    (NV_VERSION_MAJOR << 16) | (NV_VERSION_MINOR << 8) | NV_VERSION_PATCH)

// Platform
#define NV_WINDOWS        1
#define NV_LINUX          2
#define NV_APPLE          3

// Compiler
#define NV_MSVC           1
#define NV_GNUC           2
#define NV_BORLAND        3

// Endianess
#define NV_LITTLEENDIAN   1
#define NV_BIGENDIAN      2

// Bits
#define NV_32BIT          1
#define NV_64BIT          2

// Assumption
#define NV_ENDIANESS NV_LITTLEENDIAN

// Platform detection
#if defined( __WIN32__ ) || defined( _WIN32 )
#define NV_PLATFORM NV_WINDOWS
#elif defined( __APPLE_CC__)
#define NV_PLATFORM NV_APPLE
#else
#define NV_PLATFORM NV_LINUX
#endif

// Compiler detection
#if defined( _MSC_VER )
#define NV_COMPILER NV_MSVC
#define NV_COMP_VER _MSC_VER
#elif defined( __GNUC__ )
#define NV_COMPILER NV_GNUC
#define NV_COMP_VER (((__GNUC__)*100) + (__GNUC_MINOR__*10) + __GNUC_PATCHLEVEL__)
#elif defined( __BORLANDC__ )
#define NV_COMPILER NV_BORLAND
#define NV_COMP_VER 0
#else
#define NV_COMPILER 0
#define NV_COMP_VER 0
#endif

// Architecture detection
#if defined(__x86_64__) || defined(_M_X64) || defined(__powerpc64__) || defined(__alpha__)
#define NV_ARCHITECTURE NV_64BIT
#elif defined(__ia64__) || defined(__s390__) || defined(__s390x__)
#define NV_ARCHITECTURE NV_64BIT
#else
#define NV_ARCHITECTURE NV_32BIT
#endif

// Platform specific settings.
#if NV_PLATFORM == NV_WINDOWS
#ifdef _DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#ifdef DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#define NV_POSIX
#endif

#if NV_COMPILER == NV_MSVC 
#if NV_COMP_VER >= 1600
#define NV_HAS_CPP0X
#elif NV_COMP_VER >= 1200
#define NV_HAS_TR1
#else
#define NV_USE_BOOST
#endif
#endif

#if NV_COMPILER == NV_GNUC
#if NV_COMP_VER >= 460
#define NV_HAS_CPP0X
#elif NV_COMP_VER >= 400
#define NV_HAS_TR1
#else
#define NV_USE_BOOST
#endif
#endif

#include <cassert>
#define NV_ASSERT(cond, msg) assert( (cond) && msg )

#if NV_COMPILER == NV_MSVC 
#pragma warning(disable: 4201)
#endif

namespace nv
{
	class object;

	// Typedefs for fixed size types.
	typedef signed char         sint8;
	typedef signed short        sint16;
	typedef signed long         sint32;
#if NV_COMPILER == NV_MSVC
	typedef signed __int64      sint64;
#else
	typedef signed long long    sint64;
#endif

	typedef unsigned char       uint8;
	typedef unsigned short      uint16;
	typedef unsigned long       uint32;
#if NV_COMPILER == NV_MSVC
	typedef unsigned __int64    uint64;
#else
	typedef unsigned long long  uint64;
#endif

	typedef unsigned char       char8;
	typedef unsigned short      char16;
	typedef unsigned long       char32;

	typedef float  f32;
	typedef double f64;

	typedef uint64 uid;

} // namespace nv

template <typename OBJ, typename T> 
inline size_t offset_of(T OBJ::*ptr)
{
	return ((size_t)&(((OBJ*)0)->*ptr));
}

template <typename T, typename U>
inline T* down_cast(U* x)
{
#if NV_DEBUG
	T* p = dynamic_cast<T*>(x);
	if (p == 0) throw std::bad_cast();
	return p;
#else
	return static_cast<T*>(x);
#endif
}

#endif // NV_COMMON_HH
