// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/engine/animation.hh"

#include "nv/stl/range.hh"
#include "nv/lua/lua_nova.hh"
#include "nv/io/c_file_system.hh"
#include "nv/formats/nmd_loader.hh"

using namespace nv;

nv::resource< animator_data > nv::animator_manager::load_animator( const nv::string_view& id, nv::pose_data_set* poses )
{
	lua::table_guard table( m_lua, lua::path( "animators", id ) );
	return load_animator( table, id, poses );
}

bool nv::animator_manager::load_resource( nv::lua::table_guard& table, nv::shash64 id )
{
	return load_animator( table, id, nullptr ).is_valid();
}

nv::resource< animator_data > nv::animator_manager::load_animator( nv::lua::table_guard& table, nv::shash64 id, nv::pose_data_set* poses /*= nullptr */ )
{
	uint32 count = table.size();
	if ( count == 0 )
	{
		return nv::resource< animator_data >();
	}

	nv::animator_data* animator = new nv::animator_data;
	animator->id = table["id"].get_string64();
	animator->poses = poses;
	if ( poses == nullptr )
	{
		string128 poses_path = table["poses"].get_string128();
		if ( !poses_path.empty() )
		{
			nv::c_file_system fs;
			nv::stream* poses_file = open_stream( fs, poses_path );
			NV_ASSERT_ALWAYS( poses_file, "CANT FIND POSES FILE!" );
			nv::nmd_loader* ploader = new nv::nmd_loader( nullptr );
			ploader->load( *poses_file );
			delete poses_file;

			animator->poses = ploader->release_pose_data_set();
			NV_ASSERT( animator->poses, "NO POSES LOADED?" );
		}
	}

	animator->layers.resize( count );
	for ( auto layer_idx : nv::range( 1u, count ) )
	{
		nv::hash_store< shash64, uint32 > state_names;
		nv::lua::table_guard layer_table( table, layer_idx );
		nv::animator_layer_data& layer = animator->layers[layer_idx - 1];
		layer.name = layer_table["name"].get_string( m_strings );
		layer.mask = layer_table["mask"].get_sint32( -1 );
		layer.def_state = -1;

		if ( layer.mask >= 0 )
		{
			const auto& tree = animator->poses->get_tree();
			layer.mask_vector.resize( tree.size(), false );
			fill_mask_vector_rec( layer.mask_vector, tree, uint32( layer.mask ) );
		}

		if ( layer_table["states"].is_table() )
		{
			nv::lua::table_guard states_table( layer_table, "states" );
			uint32 state_count = states_table.size();
			if ( state_count > 0 )
			{
				layer.states.resize( state_count );

				// Two passes for transition name resolution
				for ( auto state_idx : nv::range( 1u, state_count ) )
				{
					nv::lua::table_guard state_table( states_table, state_idx );
					nv::animator_state_data& state = layer.states[state_idx - 1];

					state.name = state_table["name"].get_string( m_strings );
					NV_ASSERT( state.name, "Animation state without name is invalid!" );
					state_names[state.name] = state_idx - 1;
					state.loop = state_table["loop"].get_bool( true );
					state.duration = state_table["duration"].get_f32();
					state.interp = state_table["interpolation"].get_enum( nv::interpolation::SPHERICAL );

					shash64 pose_set = state_table["pose_set"].get_shash64();
					if ( pose_set )
					{
						auto it = animator->poses->m_sets.find( pose_set );
						NV_ASSERT( it->second.name, "POSE NOT FOUND" );
						uint32 pid = it->second.start;
						uint32 pose_count = it->second.count;
						uint32 start = state_table["start"].get_uint32();
						uint32 stop = state_table["stop"].get_uint32( pose_count - 1 );
						for ( nv::uint32 i = pid + start; i < pid + stop + 1; ++i )
						{
							state.poses.push_back( nv::animator_pose_data{ i/*, 0.0f*/ } );
						}

						NV_ASSERT( state.poses.size() > 0, "POSES EMPTY!" );
						state.base_pose = state.poses[0].pose;
					}
				}

				// Second pass for transitions only
				for ( auto state_idx : nv::range( 1u, state_count ) )
				{
					nv::animator_state_data& state = layer.states[state_idx - 1];
					nv::lua::table_guard state_table( states_table, state_idx );

					if ( state_table["transitions"].is_table() )
					{
						nv::lua::table_guard transitions_table( state_table, "transitions" );
						uint32 transition_count = transitions_table.size();
						if ( transition_count > 0 )
						{
							for ( auto transition_idx : nv::range( 1u, transition_count ) )
							{
								nv::lua::table_guard transition_table( transitions_table, transition_idx );
								shash64 name = transition_table["name"].get_string( m_strings );
								shash64 target_name = transition_table["target"].get_shash64();
								NV_ASSERT( name, "Transition state without name is invalid!" );
								NV_ASSERT( target_name, "Transition state without name is invalid!" );
								auto it = state_names.find( target_name );
								NV_ASSERT( it != state_names.end(), "Transition target NOT FOUND!" );

								nv::animator_transition_data tr_data;
								tr_data.target = it->second;
								tr_data.duration = transition_table["duration"].get_f32();
								tr_data.interp = transition_table["interpolation"].get_enum( nv::interpolation::SPHERICAL );
								tr_data.easing = read_easing( transition_table );

								state.transitions[name] = tr_data;
							}
						}
					}
				}
			}
		}




		shash64 def_state_name = layer_table["default"].get_shash64();
		if ( def_state_name )
		{
			auto dsi = state_names.find( def_state_name );
			NV_ASSERT( dsi != state_names.end(), "Unknown default!" );
			layer.def_state = sint32( dsi->second );
		}
	}

	return add( id, animator );
}

nv::easing animator_manager::read_easing( nv::lua::table_guard& table )
{
	nv::easing result;
	result.in = table["easing"].get_enum( nv::easing_type::NONE );
	result.in = table["ease_in"].get_enum( result.in );
	result.out = table["ease_out"].get_enum( nv::easing_type::NONE );
	if ( table[ "ease_in_out" ] )
	{
		result.in = result.out = table["ease_in_out"].get_enum( nv::easing_type::NONE );
	}
	if ( result.in == nv::easing_type::NONE && result.out == nv::easing_type::NONE )
	{
		result.in = nv::easing_type::LINEAR;
		result.out = nv::easing_type::NONE;
	}
	else
	{
		return result;
	}
	return result;
}

nv::resource< nv::animator_data > animator_manager::create_animator( const nv::string_view& id, nv::pose_data_set* poses )
{
	animator_data* data = new animator_data;
	data->poses = poses;
	return add( id, data );
}
