// Copyright (C) 2014-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, 
// see copying.txt file in root folder.

/**
* @file handle_store.hh
* @author Kornel Kisielewicz
*/

#ifndef NV_STL_HANDLE_STORE_HH
#define NV_STL_HANDLE_STORE_HH

#include <nv/common.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/handle_manager.hh>

namespace nv
{

	template < typename T, typename Handle = handle<>, typename Index = sint32 >
	class packed_indexed_array
	{
	public:
		typedef Handle               handle_type;
		typedef Index                index_type;
		typedef T                    value_type;
		typedef vector< value_type > storage_type;
		typedef typename storage_type::iterator        iterator;
		typedef typename storage_type::const_iterator  const_iterator;
		typedef typename storage_type::reference       reference;
		typedef typename storage_type::const_reference const_reference;

		packed_indexed_array() {}
		packed_indexed_array( uint32 reserve )
		{
			m_data.reserve( reserve );
			m_index.reserve( reserve );
		}

		T* insert( handle_type h )
		{
			/*index_type i = */m_index.insert( h );
			//NV_ASSERT( i == m_data.size(), "Fail!" );
			m_data.emplace_back();
			return &( m_data.back() );
		}

		bool exists( handle_type h )
		{
			return m_index.exists( h );
		}

		T* get( handle_type h )
		{
			index_type i = m_index.get( h );
			return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
		}

		const T* get( handle_type h ) const
		{
			index_type i = m_index.get( h );
			return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
		}

		void remove( handle_type h )
		{
			index_type dead_eindex = m_index.remove_swap( h );
			if ( dead_eindex == -1 ) return;
			if ( dead_eindex != static_cast<index_type>( m_data.size() - 1 ) )
			{
				m_data[unsigned( dead_eindex )] = move( m_data.back() );
			}
			m_data.pop_back();
		}

		void clear()
		{
			m_index.clear();
			m_data.clear();
		}

		const value_type& operator[] ( index_type i ) const { return m_data[i]; }
		value_type& operator[] ( index_type i ) { return m_data[i]; }
		size_t size() const { return m_data.size(); }

		iterator        begin() { return m_data.begin(); }
		const_iterator  begin()  const { return m_data.cbegin(); }
		const_iterator  cbegin() const { return m_data.cbegin(); }

		iterator        end() { return m_data.end(); }
		const_iterator  end()  const { return m_data.cend(); }
		const_iterator  cend() const { return m_data.cend(); }

	private:
		storage_type                  m_data;
		flat_index_table< handle_type, index_type > m_index;
	};

	template < typename T, typename HANDLE = handle<>, typename TINDEX = sint32 >
	class unpacked_indexed_array
	{
	public:
		typedef HANDLE              handle;
		typedef TINDEX              index_type;
		typedef vector< T >         storage;
		typedef T                   value_type;
		typedef typename storage::iterator        iterator;
		typedef typename storage::const_iterator  const_iterator;
		typedef typename storage::reference       reference;
		typedef typename storage::const_reference const_reference;

		static const index_type INVALID = index_type( -1 );

		unpacked_indexed_array() {}
		unpacked_indexed_array( uint32 reserve )
		{
			m_data.reserve( reserve );
			m_indices.reserve( reserve );
		}

		T* insert( handle h )
		{
			NV_ASSERT( !exists( h ), "Reinserting handle!" );
			resize_to( index_type( h.index() ) );
			m_indices[h.index()] = h.index();
			return &( m_data[h.index()] );
		}

		bool exists( handle h )
		{
			if ( h.is_nil() || h.index() >= m_data.size() ) return false;
			return m_indices[h.index()] != INVALID;
		}

		T* get( handle h )
		{
			if ( h.is_nil() || h.index() >= m_data.size() ) return nullptr;
			index_type i = h.index();
			return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
		}

		const T* get( handle h ) const
		{
			if ( h.is_nil() || h.index() >= m_data.size() ) return nullptr;
			index_type i = h.index();
			return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
		}

		void remove( handle h )
		{
			m_indices[h.index()] = INVALID;
		}

		void clear()
		{
			m_data.clear();
			m_indices.clear();
		}

		const value_type& operator[] ( index_type i ) const { return m_data[i]; }
		value_type& operator[] ( index_type i ) { return m_data[i]; }
		size_t size() const { return m_data.size(); }

		iterator        begin() { return m_data.begin(); }
		const_iterator  begin()  const { return m_data.cbegin(); }
		const_iterator  cbegin() const { return m_data.cbegin(); }

		iterator        end() { return m_data.end(); }
		const_iterator  end()  const { return m_data.cend(); }
		const_iterator  cend() const { return m_data.cend(); }

	private:
		void resize_to( index_type i )
		{
			index_type size = index_type( m_indices.size() );
			if ( i >= size )
			{
				if ( size == 0 ) size = 1;
				while ( i >= size ) size = size * 2;
				m_data.resize( static_cast<size_t>( size ) );
				m_indices.resize( static_cast<size_t>( size ), INVALID );
			}
		}

		vector< T >          m_data;
		vector< index_type > m_indices;
	};


	template < typename T, typename HANDLE = handle<>, typename TINDEX = sint32 >
	class handle_store
	{
	public:
		typedef HANDLE      handle;
		typedef TINDEX      index_type;
		typedef vector< T > storage;
		typedef T           value_type;
		typedef typename storage::iterator        iterator;
		typedef typename storage::const_iterator  const_iterator;
		typedef typename storage::reference       reference;
		typedef typename storage::const_reference const_reference;

		handle_store() {}

		explicit handle_store( uint32 reserve )
		{
			m_data.reserve( reserve );
		}

		handle create()
		{
			handle h = m_indexes.create_handle();
			m_data.insert( h );
			return h;
		}

		bool is_valid( handle h )
		{
			return m_indexes.is_valid( h );
		}

		const value_type* get( handle h ) const
		{
			return m_data.get( h );
		}

		value_type* get( handle h )
		{
			return m_data.get( h );
		}

		void destroy( handle e )
		{
			m_data.remove( e );
			m_indexes.free_handle( e );
		}

		void clear()
		{
			m_data.clear();
			m_indexes.clear();
		}

		const value_type& operator[] ( index_type i ) const { return m_data[unsigned( i )]; }
		value_type& operator[] ( index_type i ) { return m_data[unsigned( i )]; }
		size_t size() const { return m_data.size(); }

		iterator        begin() { return m_data.begin(); }
		const_iterator  begin()  const { return m_data.cbegin(); }
		const_iterator  cbegin() const { return m_data.cbegin(); }

		iterator        end() { return m_data.end(); }
		const_iterator  end()  const { return m_data.cend(); }
		const_iterator  cend() const { return m_data.cend(); }

	private:
		packed_indexed_array< T, handle, TINDEX > m_data;
		handle_manager< handle >                  m_indexes;
	};

}

#endif // NV_STL_HANDLE_STORE_HH
