// Copyright (C) 2015-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file growing_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief growing contiguous container handler
*/

#ifndef NV_STL_CONTAINER_GROWING_STORAGE_HH
#define NV_STL_CONTAINER_GROWING_STORAGE_HH

#include <nv/common.hh>
#include <nv/stl/container/contiguous_storage.hh>
#include <nv/stl/container/initialize_policy.hh>

namespace nv
{
	template< typename SizeType >
	struct default_next_capacity
	{
		static SizeType get( SizeType requested, SizeType capacity, SizeType max_size )
		{
			SizeType minimum = nv::max<SizeType>( capacity, 4 );
			SizeType remaining = max_size - capacity;
			if ( remaining < requested ) return 0;
			SizeType additional = nv::max( requested, capacity );
			return nv::max( minimum, remaining < additional ? max_size : capacity + additional );
		}
	};

	namespace detail
	{
		template < typename SizeType, uint32 Capacity >
		struct growing_storage_size
		{
			static constexpr uint32 max_size() { return Capacity; }
			static constexpr uint32 capacity() { return Capacity; }
			constexpr uint32 size() const { return m_size; }

			inline void size( uint32 new_size ) { m_size = static_cast< SizeType >( new_size ); }

			operator uint32( ) const { return m_size; }

			SizeType m_size = 0;
		};

		template < typename SizeType >
		struct growing_storage_size< SizeType, 0 >
		{
			static constexpr uint32 max_size() { return uint32( 0x80000000 ); }
			constexpr uint32 capacity() const { return m_capacity; }
			constexpr uint32 size() const { return m_size; }

			inline void capacity( uint32 new_capacity ) { m_capacity = static_cast< SizeType >( new_capacity ); }
			inline void size( uint32 new_size ) { m_size = static_cast< SizeType >( new_size ); }

			operator uint32() const { return m_size; }

			SizeType m_size = 0;
			SizeType m_capacity = 0;
		};
	}

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard,
		typename SizeType = uint32,
		uint32 Capacity = 0,
		typename NextCapacity = default_next_capacity< uint32 >
	>
	class growing_storage : public Storage
	{
	public:
		typedef detail::growing_storage_size< SizeType, Capacity > size_impl_type;
		typedef typename Storage::value_type value_type;
		typedef uint32                       size_type;
		typedef SizeType                     size_store_type;
		typedef value_type*                  iterator;
		typedef const value_type*            const_iterator;

		static constexpr bool is_fixed = false;

		static constexpr size_type max_size() { return size_impl_type::max_size(); }
		constexpr size_type capacity() const { return m_size.capacity(); }
		constexpr size_type size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr size_type raw_size() const { return sizeof( value_type ) * m_size; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), size() ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), size() ); }

		inline growing_storage() {}
		inline explicit growing_storage( size_type new_size ) { resize( new_size ); }
		inline explicit growing_storage( default_init ) { resize( default_init() ); }
		inline growing_storage( size_type new_size, const value_type& v ) { resize( new_size, v ); }

		// prevent copying 
		growing_storage( const growing_storage& ) = delete;
		growing_storage& operator=( const growing_storage& ) = delete;
		
		// allow move
		growing_storage( growing_storage&& other )
			: Storage( nv::move( other ) ), m_size( other.m_size )
		{
			other.m_size = size_impl_type();
		}

		template< typename InputIterator >
		growing_storage( InputIterator first, InputIterator last )
		{
			assign( first, last );
		}

		inline growing_storage& operator=( growing_storage&& other )
		{
			if ( this != &other )
			{
				m_size = other.m_size;
				Storage::operator=( nv::move( other ) );
				other.m_size = size_impl_type();
			}
			return *this;
		}

		void reserve( size_type new_capacity )
		{
			try_reserve( new_capacity, true );
		}

		void push_back( const value_type& e )
		{
			if ( try_grow( 1 ) ) copy_construct_object( Storage::data() + sint32( m_size ) - 1, e );
		}
		void push_back( value_type&& e )
		{
			if ( try_grow( 1 ) ) move_construct_object( Storage::data() + sint32( m_size ) - 1, forward<value_type>( e ) );
		}
		template < typename... Args >
		void emplace_back( Args&&... args )
		{
			if ( try_grow( 1 ) ) construct_object( Storage::data() + sint32( m_size ) - 1, forward<Args>( args )... );
		}

		void pop_back()
		{
			if ( m_size == 0 ) return;
			InitializePolicy::destroy( Storage::data() + sint32( m_size ) - 1 );
			m_size.size( m_size - 1 );
		}

		// TODO: implement swap_erase
		iterator erase( iterator position )
		{
			if ( m_size == 0 ) return iterator();
			iterator iend = Storage::data() + sint32( m_size );
			InitializePolicy::destroy( position );
			if ( ( position + 1 ) < iend )
				raw_alias_copy( position + 1, iend, position );
			m_size.size( m_size - 1 );
			return position;
		}

		iterator erase( iterator first, iterator last )
		{
			NV_ASSERT( first <= last, "Bad arguments passed to erase!" );
			iterator iend = Storage::data() + sint32( m_size );
			InitializePolicy::destroy( first, last );
			iterator position = raw_alias_copy( last, iend, first );
			m_size.size( m_size - size_type( last - first ) );
			return position;
		}

		void append( const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			NV_ASSERT( first <= last, "Bad arguments passed to append!" );
			size_type d        = size_type( distance( first, last ) );
			size_type old_size = m_size;
			if ( try_grow( d ) )
				InitializePolicy::copy( first, last, Storage::data() + old_size );
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const value_type& value )
		{
			ptrdiff_t offset = position - Storage::data();
			if ( try_grow( 1 ) )
			{
				iterator iposition = Storage::data() + offset;
				iterator iend = Storage::data() + sint32( m_size ) - 1;
				raw_alias_copy( iposition, iend, iposition + 1 );
				copy_construct_object( iposition, value );
			}
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			NV_ASSERT( first <= last, "Bad arguments passed to insert!" );
			ptrdiff_t offset = position - Storage::data();
			size_type d = size_type( distance( first, last ) );
			if ( try_grow( d ) )
			{
				iterator iposition = Storage::data() + offset;
				iterator iend = Storage::data() + sint32( m_size ) - d;
				raw_alias_copy( iposition, iend, iposition + d );
				InitializePolicy::copy( first, last, iposition );
			}
		}

		inline void resize( size_type new_size )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) InitializePolicy::initialize( Storage::data() + old_size, Storage::data() + sint32( m_size ) );
		}

		inline void resize( size_type new_size, default_init )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_construct( Storage::data() + old_size, Storage::data() + sint32( m_size ) );
		}

		inline void resize( size_type new_size, const value_type& value )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_fill( Storage::data() + old_size, Storage::data() + sint32( m_size ), value );
		}

		inline void assign( const value_type* ptr, size_type sz )
		{
			if ( ptr != nullptr && sz > 0 )
			{
				if ( try_resize( sz, false ) )
					InitializePolicy::copy( ptr, ptr + sz, Storage::data() );
			}
			else m_size.size( 0 );
		}

		template< typename InputIterator >
		inline void assign( InputIterator first, InputIterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			NV_ASSERT( first <= last, "Bad arguments passed to insert!" );
			if ( m_size > 0 ) InitializePolicy::destroy( Storage::data(), Storage::data() + sint32( m_size ) );
			size_type d = size_type( distance( first, last ) );
			if ( try_resize( d, false ) )
				InitializePolicy::copy( first, last, Storage::data() );
		}

		// explicit copy
		inline void assign( const growing_storage& other )
		{
			assign( other.data(), other.size() );
		}

		inline void clear()
		{
			if ( m_size > 0 )
			{
				InitializePolicy::destroy( Storage::data(), Storage::data() + sint32( m_size ) );
				m_size = size_impl_type();
			}
		}

		~growing_storage()
		{
			clear();
			Storage::reallocate( 0, false );
		}

	protected:

		inline void resize_impl( size_type new_size )
		{
			size_type old_size = m_size;
			if ( new_size != old_size )
			{
				if ( new_size < old_size )
				{
					InitializePolicy::destroy( Storage::data() + sint32( new_size ), Storage::data() + sint32( old_size ) );
				}
				if ( try_resize( new_size, true ) )
				{
					// TODO: error checking
				}
			}
		}

		template< uint32 C = Capacity >
		enable_if_t< C != 0, bool > try_reallocate( uint32, bool )
		{
			return false;
		}

		template< uint32 C = Capacity >
		enable_if_t< C == 0, bool > try_reallocate( uint32 new_capacity, bool copy_needed )
		{
			if ( new_capacity > 0 && Storage::reallocate( new_capacity, copy_needed ) )
			{
				m_size.capacity( new_capacity );
				return true;
			}
			else return false;
		}

		// TODO: return type error checking
		bool try_grow( uint32 amount, uint32 extra = 0 )
		{
			size_type new_size = amount + m_size + extra;
			if ( new_size > m_size.capacity() )
			{
				size_type new_capacity = NextCapacity::get( new_size - m_size.capacity(), m_size.capacity(), max_size() );
				if ( !try_reallocate( new_capacity, true ) ) return false;
			}
			m_size.size( new_size - extra );
			return true;
		}

		// TODO: return type error checking
		bool try_reserve( uint32 new_capacity, bool copy_needed )
		{
			if ( new_capacity > m_size.capacity() )
			{
				return try_reallocate( new_capacity, copy_needed );
			}
			return true;
		}
		// TODO: return type error checking
		bool try_resize( uint32 new_size, bool copy_needed, uint32 extra = 0 )
		{
			if ( new_size > m_size )
			{
				if ( try_reserve( new_size + extra, copy_needed ) )
				{
					m_size.size( new_size );
					return true;
				}
				return false;
			}
			m_size.size( new_size );
			return true;
		}
	protected:
		size_impl_type m_size;
	};

}

#endif // NV_STL_CONTAINER_GROWING_STORAGE_HH

