// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file heap.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief heap algorithms
*/

#ifndef NV_STL_ALGORITHM_HEAP_HH
#define NV_STL_ALGORITHM_HEAP_HH

#include <nv/stl/algorithm/common.hh>
#include <nv/stl/utility/common.hh>
#include <nv/stl/functional/comparisons.hh>
#include <nv/stl/iterator.hh>

namespace nv
{
	namespace detail
	{
		template< 
			typename RandomAccessIterator,
			typename T,
			typename Predicate
		>
		inline void push_heap_impl( 
			RandomAccessIterator first, 
			ptrdiff_t pivot,
			ptrdiff_t top,
			T&& value, 
			Predicate pred
		)
		{
			for ( 
				ptrdiff_t index = ( pivot - 1 ) / 2;
				top < pivot && pred( *( first + index ), value );
				index = ( pivot - 1 ) / 2 )
			{
				*( first + pivot ) = ::nv::move( *( first + index ) );
				pivot = index;
			}

			*( first + pivot ) = ::nv::move( value );
		}

		template<
			typename RandomAccessIterator,
			typename T,
			typename Predicate
		>
		inline void fix_heap(
			RandomAccessIterator first,
			ptrdiff_t pivot,
			ptrdiff_t bottom,
			T&& value,
			Predicate pred
		)
		{
			ptrdiff_t top = pivot;
			ptrdiff_t index = 2 * pivot + 2;
			for ( ; index < bottom; index = 2 * index + 2 )
			{
				if ( pred( *( first + index ), *( first + ( index - 1 ) ) ) ) --index;
				*( first + pivot ) = ::nv::move( *( first + index ) );
				pivot = index;
			}
			if ( index == bottom )
			{
				*( first + pivot ) = ::nv::move( *( first + (bottom - 1) ) );
				pivot = bottom - 1;
			}
			push_heap_impl( first, pivot, top, ::nv::move( value ), pred );
		}

		template< 
			typename RandomAccessIterator,
			typename T,
			typename Predicate
		>
		inline void pop_heap_impl( 
			RandomAccessIterator first, 
			RandomAccessIterator last,
			RandomAccessIterator dest,
			T&& value,
			Predicate pred
		)
		{
			*dest = ::nv::move( *first );
			fix_heap( first, 0, last - first, ::nv::move( value ), pred );
		}

	}

	template<
		typename RandomAccessIterator,
		typename Predicate
	>
	inline void push_heap( RandomAccessIterator first, RandomAccessIterator last, Predicate pred )
	{
		typedef iterator_traits< RandomAccessIterator > traits;
		typedef typename traits::difference_type diff_type;
		typedef typename traits::value_type value_type;
		if ( first != last )
		{
			--last;
			diff_type diff = last - first;
			if ( diff > 0 )
			{
				value_type value = ::nv::move( *last );
				detail::push_heap_impl( first, diff, 0, ::nv::move( value ), pred );
			}
		}
	}

	template< typename RandomAccessIterator >
	inline void push_heap( RandomAccessIterator first, RandomAccessIterator last )
	{
		push_heap( first, last, less<>() );
	}

	template<
		typename RandomAccessIterator,
		typename Predicate
	>
	inline void pop_heap( RandomAccessIterator first, RandomAccessIterator last, Predicate pred )
	{
		typedef iterator_traits< RandomAccessIterator > traits;
		typedef typename traits::value_type value_type;
		if ( last - first >= 2 )
		{
			value_type value = ::nv::move( *(last - 1 ) );
			detail::pop_heap_impl( first, last - 1, last - 1, ::nv::move( value ), pred );
		}
	}

	template< typename RandomAccessIterator >
	inline void pop_heap( RandomAccessIterator first, RandomAccessIterator last )
	{
		pop_heap( first, last, less<>() );
	}

	template<
		typename RandomAccessIterator,
		typename Predicate
	>
	inline void make_heap( RandomAccessIterator first, RandomAccessIterator last, Predicate pred )
	{
		typedef iterator_traits< RandomAccessIterator > traits;
		typedef typename traits::difference_type diff_type;
		typedef typename traits::value_type value_type;
		diff_type bottom = last - first;
		if ( bottom >= 2 )
		{
			for ( diff_type pivot = bottom / 2; 0 < pivot; )
			{
				--pivot;
				value_type value = ::nv::move( *( first + pivot ) );
				fix_heap( first, pivot, bottom, ::nv::move( value ), pred );
			}
		}
	}

	template< typename RandomAccessIterator >
	inline void make_heap( RandomAccessIterator first, RandomAccessIterator last )
	{
		make_heap( first, last, less<>() );
	}

	template<
		typename RandomAccessIterator,
		typename Predicate
	>
	inline void sort_heap( RandomAccessIterator first, RandomAccessIterator last, Predicate pred )
	{
		for ( ; last - first >= 2; --last )
		{
			pop_heap( first, last, pred );
		}
	}

	template< typename RandomAccessIterator >
	inline void sort_heap( RandomAccessIterator first, RandomAccessIterator last )
	{
		sort_heap( first, last, less<>() );
	}

}

#endif // NV_STL_ALGORITHM_HEAP_HH
